"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ScalarArrayDiffMissingBaseVersionStrategy = void 0;
exports.createScalarArrayDiffAlgorithm = createScalarArrayDiffAlgorithm;
var _lodash = require("lodash");
var _utility_types = require("../../../../../../../../common/utility_types");
var _prebuilt_rules = require("../../../../../../../../common/api/detection_engine/prebuilt_rules");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
/**
 * This strategy applies when all these conditions are met:
 * 1) when the base version is missing;
 * 2) and there is an update from Elastic (current version != target version);
 * 3) and the rule IS marked as customized.
 *
 * When all that is true, the scalar array diff algorithm uses this strategy
 * to determine what to do, exactly.
 */
let ScalarArrayDiffMissingBaseVersionStrategy = exports.ScalarArrayDiffMissingBaseVersionStrategy = /*#__PURE__*/function (ScalarArrayDiffMissingBaseVersionStrategy) {
  ScalarArrayDiffMissingBaseVersionStrategy["Merge"] = "Merge";
  ScalarArrayDiffMissingBaseVersionStrategy["UseTarget"] = "UseTarget";
  return ScalarArrayDiffMissingBaseVersionStrategy;
}({});
/**
 * Diff algorithm used for arrays of scalar values (eg. numbers, strings, booleans, etc.)
 *
 * NOTE: Diffing logic will be agnostic to array order
 */
function createScalarArrayDiffAlgorithm(options) {
  return function scalarArrayDiffAlgorithm(versions, isRuleCustomized) {
    const {
      base_version: baseVersion,
      current_version: currentVersion,
      target_version: targetVersion
    } = versions;
    const diffOutcome = (0, _prebuilt_rules.determineOrderAgnosticDiffOutcome)(baseVersion, currentVersion, targetVersion);
    const valueCanUpdate = (0, _prebuilt_rules.determineIfValueCanUpdate)(diffOutcome);
    const hasBaseVersion = baseVersion !== _prebuilt_rules.MissingVersion;
    const {
      mergeOutcome,
      conflict,
      mergedVersion
    } = mergeVersions({
      baseVersion: hasBaseVersion ? baseVersion : undefined,
      currentVersion,
      targetVersion,
      diffOutcome,
      isRuleCustomized,
      options
    });
    return {
      has_base_version: hasBaseVersion,
      base_version: hasBaseVersion ? baseVersion : undefined,
      current_version: currentVersion,
      target_version: targetVersion,
      merged_version: mergedVersion,
      merge_outcome: mergeOutcome,
      diff_outcome: diffOutcome,
      conflict,
      has_update: valueCanUpdate
    };
  };
}
const mergeVersions = ({
  baseVersion,
  currentVersion,
  targetVersion,
  diffOutcome,
  isRuleCustomized,
  options
}) => {
  const dedupedBaseVersion = (0, _lodash.uniq)(baseVersion);
  const dedupedCurrentVersion = (0, _lodash.uniq)(currentVersion);
  const dedupedTargetVersion = (0, _lodash.uniq)(targetVersion);
  switch (diffOutcome) {
    case _prebuilt_rules.ThreeWayDiffOutcome.StockValueNoUpdate:
    case _prebuilt_rules.ThreeWayDiffOutcome.CustomizedValueNoUpdate:
    case _prebuilt_rules.ThreeWayDiffOutcome.CustomizedValueSameUpdate:
      return {
        conflict: _prebuilt_rules.ThreeWayDiffConflict.NONE,
        mergedVersion: dedupedCurrentVersion,
        mergeOutcome: _prebuilt_rules.ThreeWayMergeOutcome.Current
      };
    case _prebuilt_rules.ThreeWayDiffOutcome.StockValueCanUpdate:
      {
        return {
          conflict: _prebuilt_rules.ThreeWayDiffConflict.NONE,
          mergedVersion: dedupedTargetVersion,
          mergeOutcome: _prebuilt_rules.ThreeWayMergeOutcome.Target
        };
      }
    case _prebuilt_rules.ThreeWayDiffOutcome.CustomizedValueCanUpdate:
      {
        const merged = (0, _helpers.mergeDedupedArrays)(dedupedBaseVersion, dedupedCurrentVersion, dedupedTargetVersion);
        return {
          conflict: _prebuilt_rules.ThreeWayDiffConflict.SOLVABLE,
          mergedVersion: merged,
          mergeOutcome: _prebuilt_rules.ThreeWayMergeOutcome.Merged
        };
      }

    // Missing base versions always return target version
    // Scenario -AA is treated as AAA
    // https://github.com/elastic/kibana/issues/210358#issuecomment-2654492854
    case _prebuilt_rules.ThreeWayDiffOutcome.MissingBaseNoUpdate:
      {
        return {
          conflict: _prebuilt_rules.ThreeWayDiffConflict.NONE,
          mergedVersion: targetVersion,
          mergeOutcome: _prebuilt_rules.ThreeWayMergeOutcome.Target
        };
      }

    // If the rule is customized, we return a SOLVABLE conflict with a merged outcome
    // Otherwise we treat scenario -AB as AAB
    // https://github.com/elastic/kibana/issues/210358#issuecomment-2654492854
    case _prebuilt_rules.ThreeWayDiffOutcome.MissingBaseCanUpdate:
      {
        if (!isRuleCustomized) {
          return {
            mergedVersion: targetVersion,
            mergeOutcome: _prebuilt_rules.ThreeWayMergeOutcome.Target,
            conflict: _prebuilt_rules.ThreeWayDiffConflict.NONE
          };
        }
        switch (options.missingBaseVersionStrategy) {
          case ScalarArrayDiffMissingBaseVersionStrategy.Merge:
            {
              return {
                mergedVersion: (0, _lodash.union)(dedupedCurrentVersion, dedupedTargetVersion),
                mergeOutcome: _prebuilt_rules.ThreeWayMergeOutcome.Merged,
                conflict: _prebuilt_rules.ThreeWayDiffConflict.SOLVABLE
              };
            }
          case ScalarArrayDiffMissingBaseVersionStrategy.UseTarget:
            {
              return {
                mergedVersion: targetVersion,
                mergeOutcome: _prebuilt_rules.ThreeWayMergeOutcome.Target,
                conflict: _prebuilt_rules.ThreeWayDiffConflict.SOLVABLE
              };
            }
          default:
            {
              return (0, _utility_types.assertUnreachable)(options.missingBaseVersionStrategy);
            }
        }
      }
    default:
      return (0, _utility_types.assertUnreachable)(diffOutcome);
  }
};