"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.reviewRuleInstallationHandler = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _utils = require("../../../routes/utils");
var _convert_prebuilt_rule_asset_to_rule_response = require("../../../rule_management/logic/detection_rules_client/converters/convert_prebuilt_rule_asset_to_rule_response");
var _prebuilt_rule_assets_client = require("../../logic/rule_assets/prebuilt_rule_assets_client");
var _prebuilt_rule_objects_client = require("../../logic/rule_objects/prebuilt_rule_objects_client");
var _utils2 = require("../../logic/utils");
var _object_case_converters = require("../../../../../utils/object_case_converters");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const reviewRuleInstallationHandler = async (context, request, response, logger) => {
  const siemResponse = (0, _utils.buildSiemResponse)(response);
  const requestParameters = (0, _object_case_converters.convertObjectKeysToCamelCase)(request.body);
  logger.debug(`reviewRuleInstallationHandler: Executing handler with params: page=${requestParameters.page}, perPage=${requestParameters.perPage}, sort=${JSON.stringify(requestParameters.sort)}, filter=${JSON.stringify(requestParameters.filter)}`);
  try {
    const ctx = await context.resolve(['core', 'alerting', 'securitySolution']);
    const soClient = ctx.core.savedObjects.client;
    const rulesClient = await ctx.alerting.getRulesClient();
    const ruleAssetsClient = (0, _prebuilt_rule_assets_client.createPrebuiltRuleAssetsClient)(soClient);
    const ruleObjectsClient = (0, _prebuilt_rule_objects_client.createPrebuiltRuleObjectsClient)(rulesClient);
    const mlAuthz = ctx.securitySolution.getMlAuthz();
    const installedRuleVersions = await ruleObjectsClient.fetchInstalledRuleVersions();
    logger.debug(`reviewRuleInstallationHandler: Found ${installedRuleVersions.length} currently installed prebuilt rules`);
    const installedRuleVersionsMap = new Map(installedRuleVersions.map(version => [version.rule_id, version]));
    const [rules, stats] = await Promise.all([fetchRules({
      ruleAssetsClient,
      logger,
      mlAuthz,
      installedRuleVersionsMap,
      requestParameters
    }), fetchStats({
      ruleAssetsClient,
      logger,
      mlAuthz,
      installedRuleVersionsMap
    })]);
    const body = {
      page: requestParameters.page,
      per_page: requestParameters.perPage,
      rules: rules.rules,
      total: rules.total,
      // Number of rules matching the filter
      stats: {
        tags: stats.tags,
        num_rules_to_install: stats.numRulesToInstall // Number of installable rules without applying filters
      }
    };
    logger.debug(`reviewRuleInstallationHandler: Returning response with total=${body.total}, num_rules_to_install=${body.stats.num_rules_to_install}`);
    return response.ok({
      body
    });
  } catch (err) {
    logger.error(`reviewRuleInstallationHandler: Caught error`, err);
    const error = (0, _securitysolutionEsUtils.transformError)(err);
    return siemResponse.error({
      body: error.message,
      statusCode: error.statusCode
    });
  }
};
exports.reviewRuleInstallationHandler = reviewRuleInstallationHandler;
async function fetchRules({
  ruleAssetsClient,
  logger,
  mlAuthz,
  installedRuleVersionsMap,
  requestParameters
}) {
  const {
    sort,
    filter,
    page,
    perPage
  } = requestParameters;
  const installableVersions = await getInstallableRuleVersions(ruleAssetsClient, logger, mlAuthz, installedRuleVersionsMap, sort, filter);
  const installableVersionsPage = installableVersions.slice((page - 1) * perPage, page * perPage);
  const installableRuleAssetsPage = await ruleAssetsClient.fetchAssetsByVersion(installableVersionsPage);
  return {
    rules: installableRuleAssetsPage.map(prebuiltRuleAsset => (0, _convert_prebuilt_rule_asset_to_rule_response.convertPrebuiltRuleAssetToRuleResponse)(prebuiltRuleAsset)),
    total: installableVersions.length
  };
}
async function fetchStats({
  ruleAssetsClient,
  logger,
  mlAuthz,
  installedRuleVersionsMap
}) {
  const installableVersionsWithoutFilter = await getInstallableRuleVersions(ruleAssetsClient, logger, mlAuthz, installedRuleVersionsMap);
  const tags = await ruleAssetsClient.fetchTagsByVersion(installableVersionsWithoutFilter);
  return {
    tags,
    numRulesToInstall: installableVersionsWithoutFilter.length
  };
}
async function getInstallableRuleVersions(ruleAssetsClient, logger, mlAuthz, installedRuleVersionsMap, sort, filter) {
  const latestRuleVersions = await ruleAssetsClient.fetchLatestVersions({
    sort,
    filter
  });
  logger.debug(`reviewRuleInstallationHandler: Fetched ${latestRuleVersions.length} latest rule versions from assets`);
  const nonInstalledLatestRuleVersions = latestRuleVersions.filter(latestVersion => !installedRuleVersionsMap.has(latestVersion.rule_id));
  logger.debug(`reviewRuleInstallationHandler: ${nonInstalledLatestRuleVersions.length} rules remaining after filtering installed rules`);
  const installableRuleVersions = await (0, _utils2.excludeLicenseRestrictedRules)(nonInstalledLatestRuleVersions, mlAuthz);
  logger.debug(`reviewRuleInstallationHandler: ${installableRuleVersions.length} rules remaining after checking license restrictions`);
  return installableRuleVersions;
}