"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createEventFilters = void 0;
var _uuid = require("uuid");
var _i18n = require("@kbn/i18n");
var _securitysolutionListConstants = require("@kbn/securitysolution-list-constants");
var _securitysolutionIoTsListTypes = require("@kbn/securitysolution-io-ts-list-types");
var _server = require("@kbn/core/server");
var _utils = require("../../endpoint/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const PROCESS_INTERACTIVE_ECS_FIELD = 'process.entry_leader.interactive';

/**
 * Create the Event Filter list if not exists and Create Event Filters for the Elastic Defend integration.
 */
const createEventFilters = async (logger, exceptionsClient, eventFilters, packagePolicy) => {
  if (!(eventFilters !== null && eventFilters !== void 0 && eventFilters.nonInteractiveSession)) {
    return;
  }
  try {
    const existingList = await exceptionsClient.getExceptionList({
      listId: _securitysolutionListConstants.ENDPOINT_ARTIFACT_LISTS.eventFilters.id,
      namespaceType: 'agnostic',
      id: undefined
    });
    if (existingList) {
      logger.debug(`Event Filter List with id "${_securitysolutionListConstants.ENDPOINT_ARTIFACT_LISTS.eventFilters.id}" already exists. Skipping creation.`);
    } else {
      logger.debug(`Event Filter List with id "${_securitysolutionListConstants.ENDPOINT_ARTIFACT_LISTS.eventFilters.id}" not found. Attempting creation.`);
      await exceptionsClient.createExceptionList({
        name: _securitysolutionListConstants.ENDPOINT_ARTIFACT_LISTS.eventFilters.name,
        namespaceType: 'agnostic',
        description: _securitysolutionListConstants.ENDPOINT_ARTIFACT_LISTS.eventFilters.description,
        listId: _securitysolutionListConstants.ENDPOINT_ARTIFACT_LISTS.eventFilters.id,
        type: _securitysolutionIoTsListTypes.ExceptionListTypeEnum.ENDPOINT_EVENTS,
        immutable: false,
        meta: undefined,
        tags: [],
        version: 1
      });
      logger.debug(`Successfully created Event Filter List with id "${_securitysolutionListConstants.ENDPOINT_ARTIFACT_LISTS.eventFilters.id}".`);
    }
  } catch (err) {
    if (_server.SavedObjectsErrorHelpers.isConflictError(err)) {
      logger.debug(`Event Filter List with id "${_securitysolutionListConstants.ENDPOINT_ARTIFACT_LISTS.eventFilters.id}" already exists. Skipping creation.`);
    } else {
      logger.error(`Error during Event Filter List handling: ${(0, _utils.wrapErrorIfNeeded)(err)}`);
      return;
    }
  }
  await createNonInteractiveSessionEventFilter(logger, exceptionsClient, packagePolicy);
};

/**
 * Create an Event Filter for non-interactive sessions and attach it to the policy
 */
exports.createEventFilters = createEventFilters;
const createNonInteractiveSessionEventFilter = async (logger, exceptionsClient, packagePolicy) => {
  try {
    await exceptionsClient.createExceptionListItem({
      listId: _securitysolutionListConstants.ENDPOINT_ARTIFACT_LISTS.eventFilters.id,
      description: _i18n.i18n.translate('xpack.securitySolution.fleetIntegration.elasticDefend.eventFilter.nonInteractiveSessions.description', {
        defaultMessage: 'Event filter for Cloud Security. Created by Elastic Defend integration.'
      }),
      name: _i18n.i18n.translate('xpack.securitySolution.fleetIntegration.elasticDefend.eventFilter.nonInteractiveSessions.name', {
        defaultMessage: 'Non-interactive Sessions'
      }),
      // Attach to the created policy
      tags: [`policy:${packagePolicy.id}`],
      osTypes: ['linux'],
      type: 'simple',
      namespaceType: 'agnostic',
      entries: [{
        field: PROCESS_INTERACTIVE_ECS_FIELD,
        operator: 'included',
        type: 'match',
        value: 'false'
      }],
      itemId: (0, _uuid.v4)(),
      meta: undefined,
      comments: [],
      expireTime: undefined
    });
  } catch (err) {
    logger.error(`Error creating Event Filter: ${(0, _utils.wrapErrorIfNeeded)(err)}`);
  }
};