"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAuditLogResponse = void 0;
var _common = require("@kbn/fleet-plugin/common");
var _constants = require("../../../../common/endpoint/constants");
var _utils = require("../../utils");
var _utils2 = require("./utils");
var _constants2 = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const queryOptions = {
  headers: {
    'X-elastic-product-origin': 'fleet'
  },
  ignore: [404]
};

/**
 * Used only for the deprecated `/api/endpoint/action_log/{agent_id}` legacy API route
 *
 * Use newer response action services instead
 *
 * @deprecated
 */
const getAuditLogResponse = async ({
  elasticAgentId,
  page,
  pageSize,
  startDate,
  endDate,
  context,
  logger
}) => {
  const size = Math.floor(pageSize / 2);
  const from = page <= 1 ? 0 : page * size - size + 1;
  const data = await getActivityLog({
    context,
    from,
    size,
    startDate,
    endDate,
    elasticAgentId,
    logger
  });
  return {
    page,
    pageSize,
    startDate,
    endDate,
    data
  };
};
exports.getAuditLogResponse = getAuditLogResponse;
const getActivityLog = async ({
  context,
  size,
  from,
  startDate,
  endDate,
  elasticAgentId,
  logger
}) => {
  var _actionsResult, _responsesResult, _responsesResult$body, _responsesResult$body2, _actionsResult2, _actionsResult2$body, _actionsResult2$body$;
  let actionsResult;
  let responsesResult;
  try {
    // fetch actions with matching agent_id
    const {
      actionIds,
      actionRequests
    } = await getActionRequestsResult({
      context,
      logger,
      elasticAgentId,
      startDate,
      endDate,
      size,
      from
    });
    actionsResult = actionRequests;

    // fetch responses with matching unique set of `action_id`s
    responsesResult = await getActionResponsesResult({
      actionIds: [...new Set(actionIds)],
      // de-dupe `action_id`s
      context,
      logger,
      elasticAgentId,
      startDate,
      endDate
    });
  } catch (error) {
    logger.error(error);
    throw error;
  }
  if (((_actionsResult = actionsResult) === null || _actionsResult === void 0 ? void 0 : _actionsResult.statusCode) !== 200) {
    logger.error(`Error fetching actions log for agent_id ${elasticAgentId}`);
    throw new Error(`Error fetching actions log for agent_id ${elasticAgentId}`);
  }

  // label record as `action`, `fleetAction`
  const responses = (0, _utils2.categorizeResponseResults)({
    results: (_responsesResult = responsesResult) === null || _responsesResult === void 0 ? void 0 : (_responsesResult$body = _responsesResult.body) === null || _responsesResult$body === void 0 ? void 0 : (_responsesResult$body2 = _responsesResult$body.hits) === null || _responsesResult$body2 === void 0 ? void 0 : _responsesResult$body2.hits
  });

  // label record as `response`, `fleetResponse`
  const actions = (0, _utils2.categorizeActionResults)({
    results: (_actionsResult2 = actionsResult) === null || _actionsResult2 === void 0 ? void 0 : (_actionsResult2$body = _actionsResult2.body) === null || _actionsResult2$body === void 0 ? void 0 : (_actionsResult2$body$ = _actionsResult2$body.hits) === null || _actionsResult2$body$ === void 0 ? void 0 : _actionsResult2$body$.hits
  });

  // filter out the duplicate endpoint actions that also have fleetActions
  // include endpoint actions that have no fleet actions
  const uniqueLogData = (0, _utils2.getUniqueLogData)([...responses, ...actions]);

  // sort by @timestamp in desc order, newest first
  const sortedData = getTimeSortedData(uniqueLogData);
  return sortedData;
};
const getTimeSortedData = data => {
  return data.sort((a, b) => new Date(b.item.data['@timestamp']) > new Date(a.item.data['@timestamp']) ? 1 : -1);
};
const getActionRequestsResult = async ({
  context,
  logger,
  elasticAgentId,
  startDate,
  endDate,
  size,
  from
}) => {
  const dateFilters = (0, _utils2.getDateFilters)({
    startDate,
    endDate
  });
  const baseActionFilters = [{
    term: {
      agents: elasticAgentId
    }
  }, {
    term: {
      input_type: 'endpoint'
    }
  }, {
    term: {
      type: 'INPUT_ACTION'
    }
  }];
  const actionsFilters = [...baseActionFilters, ...dateFilters];
  const esClient = (await context.core).elasticsearch.client.asInternalUser;
  const hasLogsEndpointActionsIndex = await (0, _utils.doesLogsEndpointActionsIndexExist)({
    esClient,
    logger,
    indexName: _constants.ENDPOINT_ACTIONS_INDEX
  });
  const actionsSearchQuery = {
    index: hasLogsEndpointActionsIndex ? _constants2.ACTION_REQUEST_INDICES : _common.AGENT_ACTIONS_INDEX,
    size,
    from,
    query: {
      bool: {
        filter: actionsFilters
      }
    },
    sort: [{
      '@timestamp': {
        order: 'desc'
      }
    }]
  };
  let actionRequests;
  try {
    var _actionRequests, _actionRequests$body, _actionRequests$body$, _actionRequests$body$2;
    actionRequests = await esClient.search(actionsSearchQuery, {
      ...queryOptions,
      meta: true
    });
    const actionIds = (_actionRequests = actionRequests) === null || _actionRequests === void 0 ? void 0 : (_actionRequests$body = _actionRequests.body) === null || _actionRequests$body === void 0 ? void 0 : (_actionRequests$body$ = _actionRequests$body.hits) === null || _actionRequests$body$ === void 0 ? void 0 : (_actionRequests$body$2 = _actionRequests$body$.hits) === null || _actionRequests$body$2 === void 0 ? void 0 : _actionRequests$body$2.map(e => {
      return e._index.includes(_constants.ENDPOINT_ACTIONS_DS) ? e._source.EndpointActions.action_id : e._source.action_id;
    });
    return {
      actionIds,
      actionRequests
    };
  } catch (error) {
    logger.error(error);
    throw error;
  }
};
const getActionResponsesResult = async ({
  context,
  logger,
  elasticAgentId,
  actionIds,
  startDate,
  endDate
}) => {
  const dateFilters = (0, _utils2.getDateFilters)({
    startDate,
    endDate
  });
  const baseResponsesFilter = [{
    term: {
      agent_id: elasticAgentId
    }
  }, {
    terms: {
      action_id: actionIds
    }
  }];
  const responsesFilters = [...baseResponsesFilter, ...dateFilters];
  const esClient = (await context.core).elasticsearch.client.asInternalUser;
  const hasLogsEndpointActionResponsesIndex = await (0, _utils.doesLogsEndpointActionsIndexExist)({
    esClient,
    logger,
    indexName: _constants.ENDPOINT_ACTION_RESPONSES_INDEX_PATTERN
  });
  const responsesSearchQuery = {
    index: hasLogsEndpointActionResponsesIndex ? _constants2.ACTION_RESPONSE_INDICES : _common.AGENT_ACTIONS_RESULTS_INDEX,
    size: 1000,
    query: {
      bool: {
        filter: responsesFilters
      }
    }
  };
  let actionResponses;
  try {
    actionResponses = await esClient.search(responsesSearchQuery, {
      ...queryOptions,
      meta: true
    });
  } catch (error) {
    logger.error(error);
    throw error;
  }
  return actionResponses;
};