"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DEFEND_INSIGHTS_TOOL_DESCRIPTION = exports.DEFEND_INSIGHTS_TOOL = void 0;
var _prompts = require("@langchain/core/prompts");
var _tools = require("@langchain/core/tools");
var _chains = require("langchain/chains");
var _output_parsers = require("langchain/output_parsers");
var _helpers = require("@kbn/elastic-assistant-plugin/server/lib/langchain/helpers");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _common = require("../../../../common");
var _services = require("../../../endpoint/services");
var _get_events = require("./get_events");
var _output_parsers2 = require("./output_parsers");
var _prompts2 = require("./prompts");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFEND_INSIGHTS_TOOL_DESCRIPTION = exports.DEFEND_INSIGHTS_TOOL_DESCRIPTION = 'Call this for Elastic Defend insights.';
/**
 * Returns a tool for generating Elastic Defend configuration insights
 */
const DEFEND_INSIGHTS_TOOL = exports.DEFEND_INSIGHTS_TOOL = Object.freeze({
  id: _elasticAssistantCommon.DEFEND_INSIGHTS_TOOL_ID,
  name: 'defendInsightsTool',
  // note: this description is overwritten when `getTool` is called
  // local definitions exist ../elastic_assistant/server/lib/prompt/tool_prompts.ts
  // local definitions can be overwritten by security-ai-prompt integration definitions
  description: DEFEND_INSIGHTS_TOOL_DESCRIPTION,
  sourceRegister: _common.APP_UI_ID,
  isSupported: params => {
    const {
      llm,
      request
    } = params;
    return (0, _helpers.requestHasRequiredAnonymizationParams)(request) && llm != null;
  },
  async getTool(params) {
    if (!this.isSupported(params)) return null;
    const {
      endpointIds,
      insightType,
      anonymizationFields,
      esClient,
      langChainTimeout,
      llm,
      onNewReplacements,
      replacements,
      request
    } = params;
    return new _tools.DynamicTool({
      name: 'DefendInsightsTool',
      description: params.description || DEFEND_INSIGHTS_TOOL_DESCRIPTION,
      func: async () => {
        if (llm == null) {
          throw new Error('LLM is required for Defend Insights');
        }
        const anonymizedEvents = await (0, _get_events.getAnonymizedEvents)({
          endpointIds,
          type: insightType,
          anonymizationFields,
          esClient,
          onNewReplacements,
          replacements
        });
        const eventsContextCount = anonymizedEvents.length;
        if (eventsContextCount === 0) {
          return JSON.stringify({
            eventsContextCount,
            insights: []
          }, null, 2);
        }
        const outputParser = (0, _output_parsers2.getDefendInsightsOutputParser)({
          type: insightType
        });
        const outputFixingParser = _output_parsers.OutputFixingParser.fromLLM(llm, outputParser);
        const prompt = new _prompts.PromptTemplate({
          template: `Answer the user's question as best you can:\n{format_instructions}\n{query}`,
          inputVariables: ['query'],
          partialVariables: {
            format_instructions: outputFixingParser.getFormatInstructions()
          }
        });
        const answerFormattingChain = new _chains.LLMChain({
          llm,
          prompt,
          outputKey: 'records',
          outputParser: outputFixingParser
        });
        const result = await answerFormattingChain.call({
          query: (0, _prompts2.getDefendInsightsPrompt)({
            type: insightType,
            events: anonymizedEvents
          }),
          timeout: langChainTimeout
        });
        const insights = result.records;
        await _services.securityWorkflowInsightsService.createFromDefendInsights(insights, request);
        return JSON.stringify({
          eventsContextCount,
          insights
        }, null, 2);
      },
      tags: [_elasticAssistantCommon.DEFEND_INSIGHTS_TOOL_ID]
    });
  }
});