"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TIMELINE_COLUMNS_CONFIG_KEY = void 0;
exports.getStoredTimelineColumnsConfig = getStoredTimelineColumnsConfig;
exports.setStoredTimelineColumnsConfig = setStoredTimelineColumnsConfig;
exports.timelineLocalStorageMiddleware = void 0;
var _zod = require("@kbn/zod");
var _selectors = require("../selectors");
var _actions = require("../actions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const LocalStorageColumnSettingsSchema = _zod.z.record(_zod.z.string(), _zod.z.object({
  initialWidth: _zod.z.number().optional(),
  id: _zod.z.string()
}));
const TIMELINE_COLUMNS_CONFIG_KEY = exports.TIMELINE_COLUMNS_CONFIG_KEY = 'timeline:columnsConfig';
function isUpdateColumnWidthAction(action) {
  return action.type === _actions.updateColumnWidth.type;
}

/**
 * Saves the timeline column settings to localStorage when it changes
 */
const timelineLocalStorageMiddleware = ({
  getState
}) => next => action => {
  // perform the action
  const ret = next(action);

  // Store the column config when it changes
  if (isUpdateColumnWidthAction(action)) {
    const timeline = (0, _selectors.selectTimelineById)(getState(), action.payload.id);
    const timelineColumnsConfig = timeline.columns.reduce((columnSettings, {
      initialWidth,
      id
    }) => {
      columnSettings[id] = {
        initialWidth,
        id
      };
      return columnSettings;
    }, {});
    setStoredTimelineColumnsConfig(timelineColumnsConfig);
  }
  return ret;
};
exports.timelineLocalStorageMiddleware = timelineLocalStorageMiddleware;
function getStoredTimelineColumnsConfig() {
  const storedConfigStr = localStorage.getItem(TIMELINE_COLUMNS_CONFIG_KEY);
  if (storedConfigStr) {
    try {
      return LocalStorageColumnSettingsSchema.parse(JSON.parse(storedConfigStr));
    } catch (_) {
      /* empty */
    }
  }
}
function setStoredTimelineColumnsConfig(config) {
  localStorage.setItem(TIMELINE_COLUMNS_CONFIG_KEY, JSON.stringify(config));
}