"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useUserAlertsItems = exports.buildVulnerableUserAggregationQuery = void 0;
var _react = require("react");
var _ecs_safety_helpers = require("../../../../../common/endpoint/models/ecs_safety_helpers");
var _manage_query = require("../../../../common/components/page/manage_query");
var _use_global_time = require("../../../../common/containers/use_global_time");
var _use_query = require("../../../../detections/containers/detection_engine/alerts/use_query");
var _constants = require("../../../../detections/containers/detection_engine/alerts/constants");
var _utils = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const USERS_BY_SEVERITY_AGG = 'usersBySeverity';
const defaultPagination = {
  pageCount: 0,
  currentPage: 0
};
const useUserAlertsItems = ({
  skip,
  queryId,
  signalIndexName,
  filterQuery
}) => {
  const [updatedAt, setUpdatedAt] = (0, _react.useState)(Date.now());
  const [items, setItems] = (0, _react.useState)([]);
  const [paginationData, setPaginationData] = (0, _react.useState)(defaultPagination);
  const {
    to,
    from,
    setQuery: setGlobalQuery,
    deleteQuery
  } = (0, _use_global_time.useGlobalTime)();
  const query = (0, _react.useMemo)(() => buildVulnerableUserAggregationQuery({
    from,
    to,
    currentPage: paginationData.currentPage,
    filterQuery
  }), [filterQuery, from, paginationData.currentPage, to]);
  const {
    setQuery,
    data,
    loading,
    request,
    response,
    refetch: refetchQuery
  } = (0, _use_query.useQueryAlerts)({
    query,
    indexName: signalIndexName,
    skip,
    queryName: _constants.ALERTS_QUERY_NAMES.VULNERABLE_USERS
  });
  (0, _react.useEffect)(() => {
    setQuery(query);
  }, [setQuery, paginationData.currentPage, query]);
  (0, _react.useEffect)(() => {
    if (data == null || !data.aggregations) {
      setItems([]);
    } else {
      setItems(parseUsersData(data.aggregations));
      setPaginationData(p => {
        var _data$aggregations;
        return {
          ...p,
          pageCount: (0, _utils.getPageCount)((_data$aggregations = data.aggregations) === null || _data$aggregations === void 0 ? void 0 : _data$aggregations.user_count.value)
        };
      });
    }
    setUpdatedAt(Date.now());
  }, [data]);
  const refetch = (0, _react.useCallback)(() => {
    if (!skip && refetchQuery) {
      refetchQuery();
    }
  }, [skip, refetchQuery]);
  const setPage = pageNumber => {
    setPaginationData(p => ({
      ...p,
      currentPage: pageNumber
    }));
  };
  (0, _manage_query.useQueryInspector)({
    deleteQuery,
    inspect: {
      dsl: [request],
      response: [response]
    },
    refetch,
    setQuery: setGlobalQuery,
    queryId,
    loading
  });
  return {
    items,
    isLoading: loading,
    updatedAt,
    pagination: {
      ...paginationData,
      setPage
    }
  };
};
exports.useUserAlertsItems = useUserAlertsItems;
const buildVulnerableUserAggregationQuery = ({
  from,
  to,
  currentPage,
  filterQuery
}) => {
  const fromValue = _utils.ITEMS_PER_PAGE * currentPage;
  return {
    query: {
      bool: {
        filter: [{
          term: {
            'kibana.alert.workflow_status': 'open'
          }
        }, {
          range: {
            '@timestamp': {
              gte: from,
              lte: to
            }
          }
        }, ...(filterQuery ? [filterQuery] : [])]
      }
    },
    size: 0,
    aggs: {
      user_count: {
        cardinality: {
          field: 'user.name'
        }
      },
      [USERS_BY_SEVERITY_AGG]: {
        terms: {
          size: 100,
          field: 'user.name',
          order: [{
            'critical.doc_count': 'desc'
          }, {
            'high.doc_count': 'desc'
          }, {
            'medium.doc_count': 'desc'
          }, {
            'low.doc_count': 'desc'
          }]
        },
        aggs: {
          critical: {
            filter: {
              term: {
                'kibana.alert.severity': 'critical'
              }
            }
          },
          high: {
            filter: {
              term: {
                'kibana.alert.severity': 'high'
              }
            }
          },
          medium: {
            filter: {
              term: {
                'kibana.alert.severity': 'medium'
              }
            }
          },
          low: {
            filter: {
              term: {
                'kibana.alert.severity': 'low'
              }
            }
          },
          bucketOfPagination: {
            bucket_sort: {
              from: fromValue,
              size: 4
            }
          }
        }
      }
    }
  };
};
exports.buildVulnerableUserAggregationQuery = buildVulnerableUserAggregationQuery;
function parseUsersData(rawAggregation) {
  var _rawAggregation$USERS;
  const buckets = (_rawAggregation$USERS = rawAggregation === null || rawAggregation === void 0 ? void 0 : rawAggregation[USERS_BY_SEVERITY_AGG].buckets) !== null && _rawAggregation$USERS !== void 0 ? _rawAggregation$USERS : [];
  return buckets.reduce((accumalatedAlertsByUser, currentUser) => {
    var _firstNonNullValue;
    accumalatedAlertsByUser.push({
      userName: (_firstNonNullValue = (0, _ecs_safety_helpers.firstNonNullValue)(currentUser.key)) !== null && _firstNonNullValue !== void 0 ? _firstNonNullValue : '-',
      totalAlerts: currentUser.doc_count,
      low: currentUser.low.doc_count,
      medium: currentUser.medium.doc_count,
      high: currentUser.high.doc_count,
      critical: currentUser.critical.doc_count
    });
    return accumalatedAlertsByUser;
  }, []);
}