"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateParsedContent = exports.validateFile = void 0;
var _i18n = require("@kbn/i18n");
var _asset_criticality = require("../../../../common/entity_analytics/asset_criticality");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const validateParsedContent = (data, experimentalFeatures) => {
  if (data.length === 0) {
    return {
      valid: [],
      invalid: [],
      errors: []
    };
  }
  let errorIndex = 1; // Error index starts from 1 because EuiCodeBlock line numbers start from 1
  const {
    valid,
    invalid,
    errors
  } = data.reduce((acc, row) => {
    const parsedRow = (0, _asset_criticality.parseAssetCriticalityCsvRow)(row, experimentalFeatures);
    if (parsedRow.valid) {
      acc.valid.push(row);
    } else {
      acc.invalid.push(row);
      acc.errors.push({
        message: parsedRow.error,
        index: errorIndex
      });
      errorIndex++;
    }
    return acc;
  }, {
    valid: [],
    invalid: [],
    errors: []
  });
  return {
    valid,
    invalid,
    errors
  };
};
exports.validateParsedContent = validateParsedContent;
const validateFile = (file, formatBytes) => {
  if (file.type !== '' &&
  // file.type might be an empty string on windows
  !_constants.SUPPORTED_FILE_TYPES.includes(file.type)) {
    return {
      valid: false,
      code: 'unsupported_file_type',
      errorMessage: _i18n.i18n.translate('xpack.securitySolution.entityAnalytics.assetCriticalityUploadPage.unsupportedFileTypeError', {
        defaultMessage: `Invalid file format selected. Please choose a {supportedFileExtensions} file and try again`,
        values: {
          supportedFileExtensions: _constants.SUPPORTED_FILE_EXTENSIONS.join(', ')
        }
      })
    };
  }
  if (file.size === 0) {
    return {
      valid: false,
      code: 'empty_file',
      errorMessage: _i18n.i18n.translate('xpack.securitySolution.entityAnalytics.assetCriticalityUploadPage.emptyFileErrorMessage', {
        defaultMessage: `The selected file is empty.`
      })
    };
  }
  if (file.size > _asset_criticality.CRITICALITY_CSV_MAX_SIZE_BYTES_WITH_TOLERANCE) {
    return {
      valid: false,
      code: 'file_size_exceeds_limit',
      errorMessage: _i18n.i18n.translate('xpack.securitySolution.entityAnalytics.assetCriticalityUploadPage.fileSizeExceedsLimitErrorMessage', {
        defaultMessage: 'File size {fileSize} exceeds maximum file size of {maxFileSize}',
        values: {
          fileSize: formatBytes(file.size),
          maxFileSize: formatBytes(_asset_criticality.CRITICALITY_CSV_MAX_SIZE_BYTES)
        }
      })
    };
  }
  return {
    valid: true
  };
};
exports.validateFile = validateFile;