"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useQueryAlerts = void 0;
var _lodash = require("lodash");
var _react = require("react");
var _api = require("./api");
var _use_track_http_request = require("../../../../common/lib/apm/use_track_http_request");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Wrapped `fetchMethod` hook that integrates
 * http-request monitoring using APM transactions.
 */
const useTrackedFetchMethod = (fetchMethod, queryName) => {
  const {
    startTracking
  } = (0, _use_track_http_request.useTrackHttpRequest)();
  const monitoredFetchMethod = (0, _react.useMemo)(() => {
    return async params => {
      const {
        endTracking
      } = startTracking({
        name: queryName
      });
      let result;
      try {
        result = await fetchMethod(params);
        endTracking('success');
      } catch (err) {
        endTracking(params.signal.aborted ? 'aborted' : 'error');
        throw err;
      }
      return result;
    };
  }, [fetchMethod, queryName, startTracking]);
  return monitoredFetchMethod;
};

/**
 * Hook for fetching Alerts from the Detection Engine API
 *
 * @param initialQuery query dsl object
 *
 */
const useQueryAlerts = ({
  fetchMethod = _api.fetchQueryAlerts,
  query: initialQuery,
  indexName,
  skip,
  queryName
}) => {
  const [query, setQuery] = (0, _react.useState)(initialQuery);
  const [alerts, setAlerts] = (0, _react.useState)({
    data: null,
    response: '',
    request: '',
    setQuery,
    refetch: null
  });
  const [loading, setLoading] = (0, _react.useState)(false);
  const fetchAlerts = useTrackedFetchMethod(fetchMethod, queryName);
  (0, _react.useEffect)(() => {
    let isSubscribed = true;
    const abortCtrl = new AbortController();
    const fetchData = async () => {
      try {
        setLoading(true);
        const alertResponse = await fetchAlerts({
          query,
          signal: abortCtrl.signal
        });
        if (isSubscribed) {
          var _ref;
          setAlerts({
            data: alertResponse,
            response: JSON.stringify(alertResponse, null, 2),
            // @ts-expect-error upgrade typescript v5.9.3
            request: JSON.stringify({
              index: (_ref = [indexName]) !== null && _ref !== void 0 ? _ref : [''],
              body: query
            }, null, 2),
            setQuery,
            refetch: fetchData
          });
        }
      } catch (error) {
        if (isSubscribed) {
          setAlerts({
            data: null,
            response: '',
            request: '',
            setQuery,
            refetch: fetchData
          });
        }
      }
      if (isSubscribed) {
        setLoading(false);
      }
    };
    if (!(0, _lodash.isEmpty)(query) && !skip) {
      fetchData();
    }
    if (skip) {
      setLoading(false);
      isSubscribed = false;
      abortCtrl.abort();
    }
    return () => {
      isSubscribed = false;
      abortCtrl.abort();
    };
  }, [query, indexName, skip, fetchAlerts]);
  return {
    loading,
    ...alerts
  };
};
exports.useQueryAlerts = useQueryAlerts;