"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.prepareSearchParams = void 0;
var _rule_filtering = require("../../../../../../../common/detection_engine/rule_management/rule_filtering");
var _detection_engine = require("../../../../../../../common/api/detection_engine");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * helper methods to prepare search params for bulk actions based on results of previous dry run
 * It excludes failed rules from search and perform bulk action on possible successfully edited rules
 * @param dryRunResult {@link DryRunResult} result of API _bulk_action dry_run
 * @param {string[]} selectedRuleIds - list of selected rule ids
 * @param filterOptions {@link FilterOptions} find filter
 * @returns either list of ids or KQL search query
 */
const prepareSearchParams = ({
  dryRunResult,
  ...props
}) => {
  // if selectedRuleIds present, filter out rules that failed during dry run
  if ('selectedRuleIds' in props) {
    const failedRuleIdsSet = new Set(dryRunResult === null || dryRunResult === void 0 ? void 0 : dryRunResult.ruleErrors.flatMap(({
      ruleIds
    }) => ruleIds));
    return {
      ids: props.selectedRuleIds.filter(id => !failedRuleIdsSet.has(id))
    };
  }

  // otherwise create filter that excludes failed results based on dry run errors
  let modifiedFilterOptions = {
    ...props.filterOptions
  };
  dryRunResult === null || dryRunResult === void 0 ? void 0 : dryRunResult.ruleErrors.forEach(({
    errorCode
  }) => {
    var _modifiedFilterOption, _modifiedFilterOption2, _modifiedFilterOption3, _modifiedFilterOption4;
    switch (errorCode) {
      case _detection_engine.BulkActionsDryRunErrCodeEnum.IMMUTABLE:
      case _detection_engine.BulkActionsDryRunErrCodeEnum.PREBUILT_CUSTOMIZATION_LICENSE:
        modifiedFilterOptions = {
          ...modifiedFilterOptions,
          showCustomRules: true
        };
        break;
      case _detection_engine.BulkActionsDryRunErrCodeEnum.MACHINE_LEARNING_INDEX_PATTERN:
      case _detection_engine.BulkActionsDryRunErrCodeEnum.MACHINE_LEARNING_AUTH:
        modifiedFilterOptions = {
          ...modifiedFilterOptions,
          excludeRuleTypes: [...((_modifiedFilterOption = modifiedFilterOptions.excludeRuleTypes) !== null && _modifiedFilterOption !== void 0 ? _modifiedFilterOption : []), 'machine_learning']
        };
        break;
      case _detection_engine.BulkActionsDryRunErrCodeEnum.ESQL_INDEX_PATTERN:
        modifiedFilterOptions = {
          ...modifiedFilterOptions,
          excludeRuleTypes: [...((_modifiedFilterOption2 = modifiedFilterOptions.excludeRuleTypes) !== null && _modifiedFilterOption2 !== void 0 ? _modifiedFilterOption2 : []), 'esql']
        };
        break;
      case _detection_engine.BulkActionsDryRunErrCodeEnum.THRESHOLD_RULE_TYPE_IN_SUPPRESSION:
        modifiedFilterOptions = {
          ...modifiedFilterOptions,
          excludeRuleTypes: [...((_modifiedFilterOption3 = modifiedFilterOptions.excludeRuleTypes) !== null && _modifiedFilterOption3 !== void 0 ? _modifiedFilterOption3 : []), 'threshold']
        };
        break;
      case _detection_engine.BulkActionsDryRunErrCodeEnum.UNSUPPORTED_RULE_IN_SUPPRESSION_FOR_THRESHOLD:
        modifiedFilterOptions = {
          ...modifiedFilterOptions,
          includeRuleTypes: [...((_modifiedFilterOption4 = modifiedFilterOptions.includeRuleTypes) !== null && _modifiedFilterOption4 !== void 0 ? _modifiedFilterOption4 : []), 'threshold']
        };
        break;
    }
  });
  return {
    query: (0, _rule_filtering.convertRulesFilterToKQL)(modifiedFilterOptions),
    ...(props.gapRange ? {
      gapRange: props.gapRange
    } : {})
  };
};
exports.prepareSearchParams = prepareSearchParams;