"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.transformExportDetailsToDryRunResult = exports.processDryRunResult = void 0;
var _rule_management = require("../../../../../../../common/api/detection_engine/rule_management");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * helper utility that transforms raw BulkActionResponse response to DryRunResult format
 * @param response - raw bulk_actions API response ({@link BulkActionResponse})
 * @returns dry run result ({@link DryRunResult})
 */
const processDryRunResult = response => {
  var _response$attributes$, _response$attributes$2;
  const processed = {
    succeededRulesCount: response === null || response === void 0 ? void 0 : response.attributes.summary.succeeded,
    failedRulesCount: response === null || response === void 0 ? void 0 : response.attributes.summary.failed,
    ruleErrors: (_response$attributes$ = response === null || response === void 0 ? void 0 : (_response$attributes$2 = response.attributes.errors) === null || _response$attributes$2 === void 0 ? void 0 : _response$attributes$2.map(({
      message,
      err_code: errorCode,
      rules
    }) => ({
      message,
      errorCode,
      ruleIds: rules.map(({
        id
      }) => id)
    }))) !== null && _response$attributes$ !== void 0 ? _response$attributes$ : []
  };
  return processed;
};

/**
 * transform rules export details {@link ExportRulesDetails} to dry run result format {@link DryRunResult}
 * @param details - {@link ExportRulesDetails} rules export details
 * @returns transformed to {@link DryRunResult}
 */
exports.processDryRunResult = processDryRunResult;
const transformExportDetailsToDryRunResult = details => {
  return {
    succeededRulesCount: details.exported_rules_count,
    failedRulesCount: details.missing_rules_count,
    // if there are rules that can't be exported, it means they are immutable. So we can safely put error code as immutable
    ruleErrors: details.missing_rules.length ? [{
      errorCode: _rule_management.BulkActionsDryRunErrCodeEnum.IMMUTABLE,
      message: "Prebuilt rules can't be exported.",
      ruleIds: details.missing_rules.map(({
        rule_id: ruleId
      }) => ruleId)
    }] : []
  };
};
exports.transformExportDetailsToDryRunResult = transformExportDetailsToDryRunResult;