"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useDisassociateExceptionList = void 0;
var _react = require("react");
var _api = require("../api/api");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Hook for removing an exception list reference from a rule
 *
 * @param http Kibana http service
 * @param ruleRuleId a rule_id (NOT id)
 * @param onError error callback
 * @param onSuccess success callback
 *
 */
const useDisassociateExceptionList = ({
  http,
  ruleRuleId,
  onError,
  onSuccess
}) => {
  const [isLoading, setLoading] = (0, _react.useState)(false);
  const disassociateList = (0, _react.useRef)(null);
  (0, _react.useEffect)(() => {
    let isSubscribed = true;
    const abortCtrl = new AbortController();
    const disassociateListFromRule = id => async exceptionLists => {
      try {
        if (isSubscribed) {
          setLoading(true);
          await (0, _api.patchRule)({
            ruleProperties: {
              rule_id: id,
              exceptions_list: exceptionLists
            },
            signal: abortCtrl.signal
          });
          onSuccess();
          setLoading(false);
        }
      } catch (err) {
        if (isSubscribed) {
          setLoading(false);
          onError(err);
        }
      }
    };
    disassociateList.current = disassociateListFromRule(ruleRuleId);
    return () => {
      isSubscribed = false;
      abortCtrl.abort();
    };
  }, [http, ruleRuleId, onError, onSuccess]);
  return [isLoading, disassociateList.current];
};
exports.useDisassociateExceptionList = useDisassociateExceptionList;