"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.siemMigrationsTelemetryEvents = exports.siemMigrationEventNames = void 0;
var _types = require("./types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const siemMigrationEventNames = exports.siemMigrationEventNames = {
  [_types.SiemMigrationsEventTypes.SetupConnectorSelected]: 'Connector Selected',
  [_types.SiemMigrationsEventTypes.SetupMigrationOpenNew]: 'Open new rules migration',
  [_types.SiemMigrationsEventTypes.SetupMigrationCreated]: 'Create new rules migration',
  [_types.SiemMigrationsEventTypes.SetupMigrationDeleted]: 'Migration deleted',
  [_types.SiemMigrationsEventTypes.SetupResourcesUploaded]: 'Upload rule resources',
  [_types.SiemMigrationsEventTypes.SetupMigrationOpenResources]: 'Rules Open Resources',
  [_types.SiemMigrationsEventTypes.SetupRulesQueryCopied]: 'Copy rules query',
  [_types.SiemMigrationsEventTypes.SetupMacrosQueryCopied]: 'Copy macros query',
  [_types.SiemMigrationsEventTypes.SetupLookupNameCopied]: 'Copy lookup name',
  [_types.SiemMigrationsEventTypes.StartMigration]: 'Start rule migration',
  [_types.SiemMigrationsEventTypes.StopMigration]: 'Stop rule migration',
  [_types.SiemMigrationsEventTypes.TranslatedRuleUpdate]: 'Update translated rule',
  [_types.SiemMigrationsEventTypes.TranslatedRuleInstall]: 'Install translated rule',
  [_types.SiemMigrationsEventTypes.TranslatedRuleBulkInstall]: 'Bulk install translated rules'
};
const baseResultActionSchema = {
  result: {
    type: 'keyword',
    _meta: {
      description: 'Indicates whether the action succeeded. Can be one of `success` or `failed`',
      optional: false
    }
  },
  errorMessage: {
    type: 'text',
    _meta: {
      description: 'The error message if action has failed',
      optional: true
    }
  }
};
const migrationIdSchema = {
  migrationId: {
    type: 'keyword',
    _meta: {
      description: 'SIEM migration ID',
      optional: false
    }
  }
};
const eventNameSchema = {
  eventName: {
    type: 'keyword',
    _meta: {
      description: 'The event name/description',
      optional: false
    }
  }
};

// This type ensures that the event schemas are correctly typed according to the event type

const eventSchemas = {
  // Setup Events
  [_types.SiemMigrationsEventTypes.SetupConnectorSelected]: {
    ...eventNameSchema,
    connectorType: {
      type: 'keyword',
      _meta: {
        description: 'Connector type',
        optional: false
      }
    },
    connectorId: {
      type: 'keyword',
      _meta: {
        description: 'Connector ID',
        optional: false
      }
    }
  },
  [_types.SiemMigrationsEventTypes.SetupMigrationOpenNew]: {
    ...eventNameSchema,
    isFirstMigration: {
      type: 'boolean',
      _meta: {
        description: 'Flag indicating if this is the first migration',
        optional: false
      }
    }
  },
  [_types.SiemMigrationsEventTypes.SetupMigrationOpenResources]: {
    ...migrationIdSchema,
    ...eventNameSchema,
    missingResourcesCount: {
      type: 'integer',
      _meta: {
        description: 'Number of missing resources',
        optional: false
      }
    }
  },
  [_types.SiemMigrationsEventTypes.SetupMigrationCreated]: {
    ...baseResultActionSchema,
    ...eventNameSchema,
    migrationId: {
      ...migrationIdSchema.migrationId,
      _meta: {
        ...migrationIdSchema.migrationId._meta,
        optional: true // Error case does not have the migration ID
      }
    },
    rulesCount: {
      type: 'integer',
      _meta: {
        description: 'Number of rules uploaded',
        optional: false
      }
    }
  },
  [_types.SiemMigrationsEventTypes.SetupMigrationDeleted]: {
    ...migrationIdSchema,
    ...baseResultActionSchema,
    ...eventNameSchema
  },
  [_types.SiemMigrationsEventTypes.SetupRulesQueryCopied]: {
    ...eventNameSchema,
    migrationId: {
      ...migrationIdSchema.migrationId,
      _meta: {
        ...migrationIdSchema.migrationId._meta,
        optional: true // Migration is not usually created yet when the query is copied
      }
    }
  },
  [_types.SiemMigrationsEventTypes.SetupMacrosQueryCopied]: {
    ...migrationIdSchema,
    ...eventNameSchema
  },
  [_types.SiemMigrationsEventTypes.SetupLookupNameCopied]: {
    ...migrationIdSchema,
    ...eventNameSchema
  },
  [_types.SiemMigrationsEventTypes.SetupResourcesUploaded]: {
    ...baseResultActionSchema,
    ...migrationIdSchema,
    ...eventNameSchema,
    type: {
      type: 'keyword',
      _meta: {
        description: `Resource type, can be one of 'macro' or 'lookup'`,
        optional: false
      }
    },
    count: {
      type: 'integer',
      _meta: {
        description: 'Number of resources uploaded',
        optional: false
      }
    }
  },
  [_types.SiemMigrationsEventTypes.StartMigration]: {
    ...baseResultActionSchema,
    ...migrationIdSchema,
    ...eventNameSchema,
    connectorId: {
      type: 'keyword',
      _meta: {
        description: 'Connector ID',
        optional: false
      }
    },
    skipPrebuiltRulesMatching: {
      type: 'boolean',
      _meta: {
        description: 'Flag indicating if prebuilt rules should be matched',
        optional: false
      }
    },
    isRetry: {
      type: 'boolean',
      _meta: {
        description: 'Flag indicating if this is a retry',
        optional: false
      }
    },
    retryFilter: {
      type: 'keyword',
      _meta: {
        description: 'Retry filter',
        optional: true
      }
    }
  },
  [_types.SiemMigrationsEventTypes.StopMigration]: {
    ...baseResultActionSchema,
    ...migrationIdSchema,
    ...eventNameSchema
  },
  // Translated Rule Events

  [_types.SiemMigrationsEventTypes.TranslatedRuleUpdate]: {
    ...baseResultActionSchema,
    ...migrationIdSchema,
    ...eventNameSchema,
    ruleMigrationId: {
      type: 'keyword',
      _meta: {
        description: 'Rule migration ID',
        optional: false
      }
    }
  },
  [_types.SiemMigrationsEventTypes.TranslatedRuleInstall]: {
    ...baseResultActionSchema,
    ...migrationIdSchema,
    ...eventNameSchema,
    ruleMigrationId: {
      type: 'keyword',
      _meta: {
        description: 'Rule migration ID',
        optional: false
      }
    },
    author: {
      type: 'keyword',
      _meta: {
        description: 'The source of the translated rule. Can be one of elastic` or `custom`',
        optional: false
      }
    },
    enabled: {
      type: 'boolean',
      _meta: {
        description: 'Is installed rule enabled',
        optional: false
      }
    },
    prebuiltRule: {
      _meta: {
        description: 'Matched elastic prebuilt rule details',
        optional: true
      },
      properties: {
        id: {
          type: 'keyword',
          _meta: {
            description: 'Matched elastic prebuilt rule ID',
            optional: false
          }
        },
        title: {
          type: 'keyword',
          _meta: {
            description: 'Matched elastic prebuilt rule title',
            optional: false
          }
        }
      }
    }
  },
  [_types.SiemMigrationsEventTypes.TranslatedRuleBulkInstall]: {
    ...baseResultActionSchema,
    ...migrationIdSchema,
    ...eventNameSchema,
    enabled: {
      type: 'boolean',
      _meta: {
        description: 'Are installed rules enabled',
        optional: false
      }
    },
    count: {
      type: 'integer',
      _meta: {
        description: 'Number of rules to be installed',
        optional: false
      }
    }
  }
};
const siemMigrationsTelemetryEvents = exports.siemMigrationsTelemetryEvents = Object.entries(eventSchemas).map(([key, schema]) => ({
  eventType: key,
  schema
}));