"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.generateSampleDocument = generateSampleDocument;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function generateSampleDocument(mappingProperties, sampleText) {
  const sampleDocument = {};
  Object.entries(mappingProperties).forEach(([field, mapping]) => {
    if ('type' in mapping) {
      switch (mapping.type) {
        case 'text':
          sampleDocument[field] = sampleText !== null && sampleText !== void 0 ? sampleText : `Sample text for ${field}`;
          break;
        case 'keyword':
          sampleDocument[field] = `sample-keyword-${field}`;
          break;
        case 'semantic_text':
          sampleDocument[field] = sampleText !== null && sampleText !== void 0 ? sampleText : `Sample text for ${field}`;
          break;
        case 'integer':
        case 'long':
          sampleDocument[field] = Math.floor(Math.random() * 100);
          break;
        case 'float':
        case 'double':
          sampleDocument[field] = Math.random() * 100;
          break;
        case 'boolean':
          sampleDocument[field] = Math.random() < 0.5;
          break;
        case 'date':
          sampleDocument[field] = new Date().toISOString();
          break;
        case 'geo_point':
          sampleDocument[field] = {
            lat: 40.7128,
            lon: -74.006
          };
          break;
        case 'nested':
          if (mapping.properties) {
            sampleDocument[field] = [generateSampleDocument(mapping.properties)];
          }
          break;
        case 'object':
          if (mapping.properties) {
            sampleDocument[field] = generateSampleDocument(mapping.properties);
          }
          break;
        case 'dense_vector':
          sampleDocument[field] = generateDenseVector(mapping);
          break;
        case 'sparse_vector':
          sampleDocument[field] = generateSparseVector();
          break;
        default:
          // Default to null for unhandled types
          sampleDocument[field] = null;
          break;
      }
    }
  });
  return sampleDocument;
}
function generateDenseVector(mapping, maxDisplayDims = 10) {
  var _mapping$dims;
  // Limit the dimensions for better UI display
  const dimension = Math.min((_mapping$dims = mapping === null || mapping === void 0 ? void 0 : mapping.dims) !== null && _mapping$dims !== void 0 ? _mapping$dims : 10, maxDisplayDims);

  // Generate an array of random floating-point numbers
  const denseVector = Array.from({
    length: dimension
  }, () => parseFloat((Math.random() * 10).toFixed(3)));
  if (dimension < ((mapping === null || mapping === void 0 ? void 0 : mapping.dims) || 0)) {
    denseVector.push('...');
  }
  return denseVector;
}
function generateSparseVector(numElements = 5, vectorSize = 100) {
  const sparseVector = {};
  const usedIndices = new Set();
  while (usedIndices.size < numElements) {
    // Generate a random index for the sparse vector
    const index = Math.floor(Math.random() * vectorSize);
    if (!usedIndices.has(index)) {
      sparseVector[index] = parseFloat((Math.random() * 10).toFixed(3));
      usedIndices.add(index);
    }
  }
  return sparseVector;
}