"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.setXState = exports.prepareRoutes = exports.mergeResponseContent = exports.getXsrfHeaderForMethod = exports.getVersionedHeaderParam = exports.getVersionedContentTypeString = exports.getPathParameters = exports.extractValidationSchemaFromRoute = exports.extractTags = exports.extractContentType = exports.createOpIdGenerator = exports.buildGlobalTags = exports.assignToPaths = void 0;
var _openapiTypes = require("openapi-types");
var _coreHttpServer = require("@kbn/core-http-server");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// eslint-disable-next-line import/no-extraneous-dependencies

const tagPrefix = 'oas-tag:';
const extractTag = tag => {
  if (tag.startsWith(tagPrefix)) {
    return tag.slice(tagPrefix.length);
  }
};
/**
 * Given an array of tags ([oas-tag:beep, oas-tag:boop]) will return a new array
 * with the tag prefix removed.
 */
const extractTags = tags => {
  if (!tags) return [];
  return tags.flatMap(tag => {
    const value = extractTag(tag);
    if (value) {
      return value;
    }
    return [];
  });
};

/**
 * Build the top-level tags entry based on the paths we extracted. We could
 * handle this while we are iterating over the routes, but this approach allows
 * us to keep this as a global document concern at the expense of some extra
 * processing.
 */
exports.extractTags = extractTags;
const buildGlobalTags = (paths, additionalTags = []) => {
  const tags = new Set(additionalTags);
  for (const path of Object.values(paths)) {
    for (const method of Object.values(_openapiTypes.OpenAPIV3.HttpMethods)) {
      var _path$method;
      if (path !== null && path !== void 0 && (_path$method = path[method]) !== null && _path$method !== void 0 && _path$method.tags) {
        path[method].tags.forEach(tag => tags.add(tag));
      }
    }
  }
  return Array.from(tags).sort((a, b) => a.localeCompare(b)).map(name => ({
    name
  }));
};
exports.buildGlobalTags = buildGlobalTags;
const getPathParameters = path => {
  return Array.from(path.matchAll(/\{([^{}?]+\??)\}/g)).reduce((acc, [_, key]) => {
    const optional = key.endsWith('?');
    acc[optional ? key.slice(0, key.length - 1) : key] = {
      optional
    };
    return acc;
  }, {});
};
exports.getPathParameters = getPathParameters;
const extractContentType = body => {
  if (body !== null && body !== void 0 && body.accepts) {
    return Array.isArray(body.accepts) ? body.accepts : [body.accepts];
  }
  return ['application/json'];
};
exports.extractContentType = extractContentType;
const getVersionedContentTypeString = (version, access, acceptedContentTypes) => {
  if (access === 'internal') {
    return `${acceptedContentTypes.join('; ')}; Elastic-Api-Version=${version}`;
  }
  // Exclude Elastic-Api-Version header for public routes for now, this means public routes
  // can only generate spec for one version at a time.
  return `${acceptedContentTypes.join('; ')}`;
};
exports.getVersionedContentTypeString = getVersionedContentTypeString;
const extractValidationSchemaFromRoute = route => {
  if (!route.validationSchemas) return undefined;
  return (0, _coreHttpServer.getRequestValidation)(route.validationSchemas);
};
exports.extractValidationSchemaFromRoute = extractValidationSchemaFromRoute;
const getVersionedHeaderParam = (defaultVersion, versions) => ({
  in: 'header',
  name: 'elastic-api-version',
  description: 'The version of the API to use',
  schema: {
    type: 'string',
    enum: versions,
    default: defaultVersion
  }
});
exports.getVersionedHeaderParam = getVersionedHeaderParam;
const prepareRoutes = (routes, filters) => {
  if (Object.getOwnPropertyNames(filters).length === 0) return routes;
  return routes.filter(route => {
    if (route.options.excludeFromOAS) return false;
    if (filters.excludePathsMatching && filters.excludePathsMatching.some(ex => route.path.startsWith(ex))) {
      return false;
    }
    if (filters.pathStartsWith && !filters.pathStartsWith.some(p => route.path.startsWith(p))) {
      return false;
    }
    if (filters.access === 'public' && route.options.access !== 'public') {
      return false;
    }
    if (filters.access === 'internal' && route.options.access != null && route.options.access !== 'internal') {
      return false;
    }
    return true;
  });
};
exports.prepareRoutes = prepareRoutes;
const assignToPaths = (paths, path, pathObject) => {
  const pathName = path.replace(/[\?\*]/g, '');
  paths[pathName] = {
    ...paths[pathName],
    ...pathObject
  };
};
exports.assignToPaths = assignToPaths;
const mergeResponseContent = (a, b) => {
  const mergedContent = {
    ...(a !== null && a !== void 0 ? a : {}),
    ...(b !== null && b !== void 0 ? b : {})
  };
  return {
    ...(Object.keys(mergedContent).length ? {
      content: mergedContent
    } : {})
  };
};
exports.mergeResponseContent = mergeResponseContent;
const getXsrfHeaderForMethod = (method, options) => {
  if (method === 'get' || method === 'options' || (options === null || options === void 0 ? void 0 : options.xsrfRequired) === false) return [];
  return [{
    description: 'A required header to protect against CSRF attacks',
    in: 'header',
    name: 'kbn-xsrf',
    required: true,
    schema: {
      example: 'true',
      type: 'string'
    }
  }];
};
exports.getXsrfHeaderForMethod = getXsrfHeaderForMethod;
const setXState = (availability, operation) => {
  if (availability) {
    if (availability.stability === 'experimental') {
      operation['x-state'] = 'Technical Preview';
    }
    if (availability.stability === 'beta') {
      operation['x-state'] = 'Beta';
    }
  }
};
exports.setXState = setXState;
/**
 * Best effort to generate operation IDs from route values
 */
const createOpIdGenerator = () => {
  const idMap = new Map();
  return function getOpId({
    path,
    method
  }) {
    var _idMap$get;
    if (!method || !path) {
      throw new Error(`Must provide method and path, received: method: "${method}", path: "${path}"`);
    }
    path = path.trim().replace(/^[\/]+/, '').replace(/[\/]+$/, '').toLowerCase();
    const removePrefixes = ['internal/api/', 'internal/', 'api/']; // longest to shortest
    for (const prefix of removePrefixes) {
      if (path.startsWith(prefix)) {
        path = path.substring(prefix.length);
        break;
      }
    }
    path = path.replace(/[\{\}\?\*]/g, '') // remove special chars
    .replace(/[\/_]/g, '-') // everything else to dashes
    .replace(/[-]+/g, '-'); // single dashes

    const opId = `${method.toLowerCase()}-${path}`;
    const cachedCount = (_idMap$get = idMap.get(opId)) !== null && _idMap$get !== void 0 ? _idMap$get : 0;
    idMap.set(opId, cachedCount + 1);
    return cachedCount > 0 ? `${opId}-${cachedCount + 1}` : opId;
  };
};
exports.createOpIdGenerator = createOpIdGenerator;