"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EventTracker = void 0;
var _reportingServer = require("@kbn/reporting-server");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class EventTracker {
  constructor(analytics, reportId, exportType, objectType) {
    this.analytics = analytics;
    this.reportId = reportId;
    this.exportType = exportType;
    this.objectType = objectType;
  }
  track(eventType, eventFields) {
    try {
      this.analytics.reportEvent(eventType, eventFields);
    } catch (err) {
      // eslint-disable-next-line no-console
      console.error(err);
    }
  }

  /*
   * When a request is made to generate a report,
   * track if the request came to the public API
   * (scripts / Watcher) and if the export type is
   * deprecated
   */
  createReport({
    isDeprecated,
    isPublicApi
  }) {
    this.track(_reportingServer.EventType.REPORT_CREATION, {
      [_reportingServer.FieldType.REPORT_ID]: this.reportId,
      [_reportingServer.FieldType.EXPORT_TYPE]: this.exportType,
      [_reportingServer.FieldType.OBJECT_TYPE]: this.objectType,
      [_reportingServer.FieldType.IS_DEPRECATED]: isDeprecated,
      [_reportingServer.FieldType.IS_PUBLIC_API]: isPublicApi
    });
  }

  /*
   * When a report job is claimed, the time since
   * creation equals the time spent waiting in the
   * queue.
   */
  claimJob(opts) {
    const {
      timeSinceCreation
    } = opts;
    this.track(_reportingServer.EventType.REPORT_CLAIM, {
      [_reportingServer.FieldType.REPORT_ID]: this.reportId,
      [_reportingServer.FieldType.EXPORT_TYPE]: this.exportType,
      [_reportingServer.FieldType.OBJECT_TYPE]: this.objectType,
      [_reportingServer.FieldType.DURATION_MS]: timeSinceCreation
    });
  }

  /*
   * When a report job is completed, the time since
   * creation equals the time spent waiting in queue +
   * retries + executing the final report.
   */
  completeJobScreenshot(opts) {
    const {
      byteSize,
      timeSinceCreation,
      numPages,
      screenshotLayout,
      screenshotPixels
    } = opts;
    this.track(_reportingServer.EventType.REPORT_COMPLETION_SCREENSHOT, {
      [_reportingServer.FieldType.REPORT_ID]: this.reportId,
      [_reportingServer.FieldType.EXPORT_TYPE]: this.exportType,
      [_reportingServer.FieldType.OBJECT_TYPE]: this.objectType,
      [_reportingServer.FieldType.DURATION_MS]: timeSinceCreation,
      [_reportingServer.FieldType.BYTE_SIZE]: byteSize,
      [_reportingServer.FieldType.NUM_PAGES]: numPages,
      [_reportingServer.FieldType.SCREENSHOT_LAYOUT]: screenshotLayout,
      [_reportingServer.FieldType.SCREENSHOT_PIXELS]: screenshotPixels
    });
  }
  completeJobCsv(opts) {
    const {
      byteSize,
      timeSinceCreation,
      csvRows
    } = opts;
    this.track(_reportingServer.EventType.REPORT_COMPLETION_CSV, {
      [_reportingServer.FieldType.REPORT_ID]: this.reportId,
      [_reportingServer.FieldType.EXPORT_TYPE]: this.exportType,
      [_reportingServer.FieldType.OBJECT_TYPE]: this.objectType,
      [_reportingServer.FieldType.DURATION_MS]: timeSinceCreation,
      [_reportingServer.FieldType.BYTE_SIZE]: byteSize,
      [_reportingServer.FieldType.CSV_ROWS]: csvRows
    });
  }

  /*
   * When a report job fails, the time since creation
   * equals the time spent waiting in queue + time
   * spent on retries + the time spent attempting
   * execution
   */
  failJob(opts) {
    const {
      timeSinceCreation,
      errorMessage,
      errorCode
    } = opts;
    this.track(_reportingServer.EventType.REPORT_ERROR, {
      [_reportingServer.FieldType.REPORT_ID]: this.reportId,
      [_reportingServer.FieldType.EXPORT_TYPE]: this.exportType,
      [_reportingServer.FieldType.OBJECT_TYPE]: this.objectType,
      [_reportingServer.FieldType.DURATION_MS]: timeSinceCreation,
      [_reportingServer.FieldType.ERROR_MESSAGE]: errorMessage,
      [_reportingServer.FieldType.ERROR_CODE]: errorCode
    });
  }

  /*
   * When a report job is downloaded, we want to
   * know how old the job is
   */
  downloadReport(opts) {
    const {
      timeSinceCreation
    } = opts;
    this.track(_reportingServer.EventType.REPORT_DOWNLOAD, {
      [_reportingServer.FieldType.REPORT_ID]: this.reportId,
      [_reportingServer.FieldType.EXPORT_TYPE]: this.exportType,
      [_reportingServer.FieldType.OBJECT_TYPE]: this.objectType,
      [_reportingServer.FieldType.DURATION_MS]: timeSinceCreation
    });
  }

  /*
   * When a report job is deleted, we want to know
   * how old the job is, and what type of error it
   * may have had
   */
  deleteReport(opts) {
    const {
      timeSinceCreation
    } = opts;
    this.track(_reportingServer.EventType.REPORT_DELETION, {
      [_reportingServer.FieldType.REPORT_ID]: this.reportId,
      [_reportingServer.FieldType.EXPORT_TYPE]: this.exportType,
      [_reportingServer.FieldType.OBJECT_TYPE]: this.objectType,
      [_reportingServer.FieldType.DURATION_MS]: timeSinceCreation
    });
  }
}
exports.EventTracker = EventTracker;