"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RunSingleReportTask = void 0;
var _moment = _interopRequireDefault(require("moment"));
var _reportingCommon = require("@kbn/reporting-common");
var _ = require(".");
var _map_to_reporting_error = require("../../../common/errors/map_to_reporting_error");
var _store = require("../store");
var _error_logger = require("./error_logger");
var _run_report = require("./run_report");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class RunSingleReportTask extends _run_report.RunReportTask {
  get TYPE() {
    return _.REPORTING_EXECUTE_TYPE;
  }
  async claimJob(task) {
    const store = await this.opts.reporting.getStore();
    const report = await store.findReportFromTask(task); // receives seq_no and primary_term

    if (report.status === 'completed') {
      throw new Error(`Can not claim the report job: it is already completed!`);
    }
    const m = (0, _moment.default)();

    // check if job has exceeded the configured maxAttempts
    const maxAttempts = this.getMaxAttempts();
    if (report.attempts >= maxAttempts) {
      let err;
      if (report.error && (0, _map_to_reporting_error.isExecutionError)(report.error)) {
        // We have an error stored from a previous attempts, so we'll use that
        // error to fail the job and return it to the user.
        const {
          error
        } = report;
        err = (0, _map_to_reporting_error.mapToReportingError)(error);
        err.stack = error.stack;
      } else {
        if (report.error && report.error instanceof Error) {
          (0, _error_logger.errorLogger)(this.logger, 'Error executing report', report.error);
        }
        err = new _reportingCommon.QueueTimeoutError(`Max attempts reached (${maxAttempts}). Queue timeout reached.`);
      }
      await this.failJob(report, err);
      throw err;
    }
    const startTime = m.toISOString();
    const expirationTime = m.add(this.queueTimeout).toISOString();
    const doc = {
      kibana_id: this.kibanaId,
      kibana_name: this.kibanaName,
      attempts: report.attempts + 1,
      max_attempts: maxAttempts,
      started_at: startTime,
      timeout: this.queueTimeout,
      process_expiration: expirationTime
    };
    const claimedReport = new _store.SavedReport({
      ...report,
      ...doc
    });
    this.logger.info(`Claiming ${claimedReport.jobtype} ${report._id} ` + `[_index: ${report._index}] ` + `[_seq_no: ${report._seq_no}] ` + `[_primary_term: ${report._primary_term}] ` + `[attempts: ${report.attempts}] ` + `[process_expiration: ${expirationTime}]`, {
      tags: [report._id]
    });

    // event tracking of claimed job
    const eventTracker = this.getEventTracker(report);
    const timeSinceCreation = Date.now() - new Date(report.created_at).valueOf();
    eventTracker === null || eventTracker === void 0 ? void 0 : eventTracker.claimJob({
      timeSinceCreation
    });
    const resp = await store.setReportClaimed(claimedReport, doc);
    claimedReport._seq_no = resp._seq_no;
    claimedReport._primary_term = resp._primary_term;
    return claimedReport;
  }
  async prepareJob(taskInstance) {
    const {
      attempts: taskAttempts,
      params: reportTaskParams
    } = taskInstance;
    let report;
    const isLastAttempt = taskAttempts >= this.getMaxAttempts();

    // find the job in the store and set status to processing
    const task = reportTaskParams;
    const jobId = task === null || task === void 0 ? void 0 : task.id;
    try {
      if (!jobId) {
        throw new Error('Invalid report data provided in scheduled task!');
      }

      // Update job status to claimed
      report = await this.claimJob(task);
    } catch (failedToClaim) {
      // error claiming report - log the error
      // could be version conflict, or too many attempts or no longer connected to ES
      (0, _error_logger.errorLogger)(this.logger, `Error in claiming ${jobId}`, failedToClaim);
    }
    return {
      isLastAttempt,
      jobId,
      report,
      task
    };
  }
  getMaxAttempts() {
    var _this$opts$config$cap;
    return (_this$opts$config$cap = this.opts.config.capture.maxAttempts) !== null && _this$opts$config$cap !== void 0 ? _this$opts$config$cap : 1;
  }
  async notify() {}
  getTaskDefinition() {
    const queueTimeout = this.getQueueTimeout();
    const maxConcurrency = this.getMaxConcurrency();
    const maxAttempts = this.getMaxAttempts();
    return {
      type: _.REPORTING_EXECUTE_TYPE,
      title: 'Reporting: execute job',
      createTaskRunner: this.getTaskRunner(),
      maxAttempts: maxAttempts + 1,
      // Add 1 so we get an extra attempt in case of failure during a Kibana restart
      timeout: queueTimeout,
      maxConcurrency
    };
  }
  async scheduleTask(request, params) {
    const reportingHealth = await this.opts.reporting.getHealthInfo();
    const shouldScheduleWithApiKey = reportingHealth.hasPermanentEncryptionKey && reportingHealth.isSufficientlySecure;
    const taskInstance = {
      taskType: _.REPORTING_EXECUTE_TYPE,
      state: {},
      params
    };
    return shouldScheduleWithApiKey ? await this.getTaskManagerStart().schedule(taskInstance, {
      request
    }) : await this.getTaskManagerStart().schedule(taskInstance);
  }
}
exports.RunSingleReportTask = RunSingleReportTask;