"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RunReportTask = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _moment = _interopRequireDefault(require("moment"));
var _rxjs = _interopRequireWildcard(require("rxjs"));
var Rx = _rxjs;
var _reportingCommon = require("@kbn/reporting-common");
var _reportingServer = require("@kbn/reporting-server");
var _server = require("@kbn/task-manager-plugin/server");
var _coreHttpServerUtils = require("@kbn/core-http-server-utils");
var _common = require("@kbn/spaces-plugin/common");
var _map_to_reporting_error = require("../../../common/errors/map_to_reporting_error");
var _ = require(".");
var _store = require("../store");
var _error_logger = require("./error_logger");
var _content_stream = require("../content_stream");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function isOutput(output) {
  return output.size != null;
}
function parseError(error) {
  if (error instanceof Error) {
    return {
      name: error.constructor.name,
      message: error.message,
      stack: error.stack,
      cause: error.cause
    };
  }
  return error;
}
class RunReportTask {
  constructor(opts) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "queueTimeout", void 0);
    (0, _defineProperty2.default)(this, "taskManagerStart", void 0);
    (0, _defineProperty2.default)(this, "kibanaId", void 0);
    (0, _defineProperty2.default)(this, "kibanaName", void 0);
    (0, _defineProperty2.default)(this, "exportTypesRegistry", void 0);
    (0, _defineProperty2.default)(this, "eventTracker", void 0);
    (0, _defineProperty2.default)(this, "emailNotificationService", void 0);
    this.opts = opts;
    this.logger = opts.logger.get('runTask');
    this.exportTypesRegistry = opts.reporting.getExportTypesRegistry();
    this.queueTimeout = (0, _reportingCommon.durationToNumber)(opts.config.queue.timeout);
  }

  // Abstract methods

  // Public methods
  async init(taskManager, emailNotificationService) {
    this.taskManagerStart = taskManager;
    const {
      uuid,
      name
    } = this.opts.reporting.getServerInfo();
    this.kibanaId = uuid;
    this.kibanaName = name;
    this.emailNotificationService = emailNotificationService;
  }
  getStatus() {
    if (this.taskManagerStart) {
      return _.ReportingTaskStatus.INITIALIZED;
    }
    return _.ReportingTaskStatus.UNINITIALIZED;
  }

  // Protected methods
  getTaskManagerStart() {
    if (!this.taskManagerStart) {
      throw new Error('Reporting task runner has not been initialized!');
    }
    return this.taskManagerStart;
  }
  getEventTracker(report) {
    if (this.eventTracker) {
      return this.eventTracker;
    }
    const eventTracker = this.opts.reporting.getEventTracker(report._id, report.jobtype, report.payload.objectType);
    this.eventTracker = eventTracker;
    return this.eventTracker;
  }
  getJobContentEncoding(jobType) {
    const exportType = this.exportTypesRegistry.getByJobType(jobType);
    return exportType.jobContentEncoding;
  }
  getJobContentExtension(jobType) {
    const exportType = this.exportTypesRegistry.getByJobType(jobType);
    return exportType.jobContentExtension;
  }
  getMaxConcurrency() {
    return this.opts.config.queue.pollEnabled ? 1 : 0;
  }
  getQueueTimeout() {
    // round up from ms to the nearest second
    return Math.ceil((0, _reportingCommon.numberToDuration)(this.opts.config.queue.timeout).asSeconds()) + 's';
  }
  async failJob(report, error) {
    var _docOutput, _docOutput$error_code, _docOutput2, _error$message;
    const message = `Failing ${report.jobtype} job ${report._id}`;
    const logger = this.logger.get(report._id);

    // log the error
    let docOutput;
    if (error) {
      (0, _error_logger.errorLogger)(logger, message, error, [report._id]);
      docOutput = this.formatOutput(error);
    } else {
      (0, _error_logger.errorLogger)(logger, message);
    }

    // update the report in the store
    const store = await this.opts.reporting.getStore();
    const completedTime = (0, _moment.default)();
    const doc = {
      completed_at: completedTime.toISOString(),
      output: (_docOutput = docOutput) !== null && _docOutput !== void 0 ? _docOutput : null
    };

    // event tracking of failed job
    const eventTracker = this.getEventTracker(report);
    const timeSinceCreation = Date.now() - new Date(report.created_at).valueOf();
    eventTracker === null || eventTracker === void 0 ? void 0 : eventTracker.failJob({
      timeSinceCreation,
      errorCode: (_docOutput$error_code = (_docOutput2 = docOutput) === null || _docOutput2 === void 0 ? void 0 : _docOutput2.error_code) !== null && _docOutput$error_code !== void 0 ? _docOutput$error_code : 'unknown',
      errorMessage: (_error$message = error === null || error === void 0 ? void 0 : error.message) !== null && _error$message !== void 0 ? _error$message : 'unknown'
    });
    return await store.setReportFailed(report, doc);
  }
  async saveExecutionError(report, failedToExecuteErr) {
    const message = `Saving execution error for ${report.jobtype} job ${report._id}`;
    const errorParsed = parseError(failedToExecuteErr);
    const logger = this.logger.get(report._id);
    // log the error
    (0, _error_logger.errorLogger)(logger, message, failedToExecuteErr);

    // update the report in the store
    const store = await this.opts.reporting.getStore();
    const doc = {
      output: null,
      error: errorParsed
    };
    return await store.setReportError(report, doc);
  }
  async saveExecutionWarning(report, output, message) {
    this.logger.warn(message, {
      tags: [report._id]
    });

    // update the report in the store
    const store = await this.opts.reporting.getStore();
    const doc = {
      output,
      warning: message
    };
    return await store.setReportWarning(report, doc);
  }
  formatOutput(output) {
    const docOutput = {};
    const unknownMime = null;
    if (isOutput(output)) {
      docOutput.content_type = output.content_type || unknownMime;
      docOutput.max_size_reached = output.max_size_reached;
      docOutput.csv_contains_formulas = output.csv_contains_formulas;
      docOutput.size = output.size;
      docOutput.warnings = output.warnings && output.warnings.length > 0 ? output.warnings : undefined;
      docOutput.error_code = output.error_code;
    } else {
      var _output$humanFriendly;
      const defaultOutput = null;
      docOutput.content = ((_output$humanFriendly = output.humanFriendlyMessage) === null || _output$humanFriendly === void 0 ? void 0 : _output$humanFriendly.call(output)) || output.toString() || defaultOutput;
      docOutput.content_type = unknownMime;
      docOutput.warnings = [output.toString()];
      docOutput.error_code = output.code;
      docOutput.size = typeof docOutput.content === 'string' ? docOutput.content.length : 0;
    }
    return docOutput;
  }
  async getRequestToUse({
    requestFromTask,
    spaceId,
    encryptedHeaders
  }) {
    let useApiKeyAuthentication = false;
    let apiKeyAuthHeaders;
    if (requestFromTask) {
      apiKeyAuthHeaders = requestFromTask.headers;
      useApiKeyAuthentication = true;
      this.logger.debug(`Using API key authentication with request from task instance`);
    }
    let decryptedHeaders;
    if (this.opts.config.encryptionKey && encryptedHeaders) {
      // get decrypted headers
      decryptedHeaders = await (0, _reportingServer.decryptJobHeaders)(this.opts.config.encryptionKey, encryptedHeaders, this.logger);
    }
    if (!decryptedHeaders && !apiKeyAuthHeaders) {
      throw new _reportingCommon.MissingAuthenticationError();
    }
    let headersToUse = {};
    if (useApiKeyAuthentication && apiKeyAuthHeaders) {
      this.logger.debug(`Merging API key authentication headers with decrypted headers`);
      const {
        cookie,
        authorization,
        ...restDecryptedHeaders
      } = decryptedHeaders || {};
      headersToUse = {
        ...apiKeyAuthHeaders,
        ...restDecryptedHeaders
      };
    } else {
      this.logger.debug(`Using decrypted headers only`);
      headersToUse = decryptedHeaders || {};
    }
    return this.getFakeRequest(headersToUse, spaceId, this.logger);
  }
  getFakeRequest(headers, spaceId, logger = this.logger) {
    var _setupDeps$spaces;
    const rawRequest = {
      headers,
      path: '/'
    };
    const fakeRequest = (0, _coreHttpServerUtils.kibanaRequestFactory)(rawRequest);
    const setupDeps = this.opts.reporting.getPluginSetupDeps();
    const spacesService = (_setupDeps$spaces = setupDeps.spaces) === null || _setupDeps$spaces === void 0 ? void 0 : _setupDeps$spaces.spacesService;
    if (spacesService) {
      if (spaceId && spaceId !== _common.DEFAULT_SPACE_ID) {
        logger.info(`Generating request for space: ${spaceId}`);
        setupDeps.basePath.set(fakeRequest, `/s/${spaceId}`);
      }
    }
    return fakeRequest;
  }
  async performJob({
    task,
    fakeRequest,
    taskInstanceFields,
    cancellationToken,
    stream
  }) {
    const exportType = this.exportTypesRegistry.getByJobType(task.jobtype);
    if (!exportType) {
      throw new Error(`No export type from ${task.jobtype} found to execute report`);
    }
    // run the report
    // if workerFn doesn't finish before timeout, call the cancellationToken and throw an error
    const request = await this.getRequestToUse({
      requestFromTask: fakeRequest,
      spaceId: task.payload.spaceId,
      encryptedHeaders: task.payload.headers
    });
    return Rx.lastValueFrom(Rx.from(exportType.runTask({
      jobId: task.id,
      payload: task.payload,
      request,
      taskInstanceFields,
      cancellationToken,
      stream
    })).pipe((0, _rxjs.timeout)(this.queueTimeout)) // throw an error if a value is not emitted before timeout
    );
  }
  async completeJob(report, output) {
    var _output$metrics, _output$metrics2, _output$metrics3;
    let docId = `/${report._index}/_doc/${report._id}`;
    this.logger.debug(`Saving ${report.jobtype} to ${docId}.`, {
      tags: [report._id]
    });
    const completedTime = (0, _moment.default)();
    const docOutput = this.formatOutput(output);
    const store = await this.opts.reporting.getStore();
    const doc = {
      completed_at: completedTime.toISOString(),
      metrics: output.metrics,
      output: docOutput
    };
    docId = `/${report._index}/_doc/${report._id}`;
    const resp = await store.setReportCompleted(report, doc);
    this.logger.info(`Saved ${report.jobtype} job ${docId}`, {
      tags: [report._id]
    });
    report._seq_no = resp._seq_no;
    report._primary_term = resp._primary_term;

    // event tracking of completed job
    const eventTracker = this.getEventTracker(report);
    const byteSize = docOutput.size;
    const timeSinceCreation = completedTime.valueOf() - new Date(report.created_at).valueOf();
    if (((_output$metrics = output.metrics) === null || _output$metrics === void 0 ? void 0 : _output$metrics.csv) != null) {
      var _output$metrics$csv$r;
      eventTracker === null || eventTracker === void 0 ? void 0 : eventTracker.completeJobCsv({
        byteSize,
        timeSinceCreation,
        csvRows: (_output$metrics$csv$r = output.metrics.csv.rows) !== null && _output$metrics$csv$r !== void 0 ? _output$metrics$csv$r : -1
      });
    } else if (((_output$metrics2 = output.metrics) === null || _output$metrics2 === void 0 ? void 0 : _output$metrics2.pdf) != null || ((_output$metrics3 = output.metrics) === null || _output$metrics3 === void 0 ? void 0 : _output$metrics3.png) != null) {
      var _report$payload$layou, _report$payload$layou2, _report$payload$layou3, _report$payload$layou4, _output$metrics$pdf$p, _output$metrics$pdf;
      const {
        width,
        height
      } = (_report$payload$layou = (_report$payload$layou2 = report.payload.layout) === null || _report$payload$layou2 === void 0 ? void 0 : _report$payload$layou2.dimensions) !== null && _report$payload$layou !== void 0 ? _report$payload$layou : {};
      eventTracker === null || eventTracker === void 0 ? void 0 : eventTracker.completeJobScreenshot({
        byteSize,
        timeSinceCreation,
        screenshotLayout: (_report$payload$layou3 = (_report$payload$layou4 = report.payload.layout) === null || _report$payload$layou4 === void 0 ? void 0 : _report$payload$layou4.id) !== null && _report$payload$layou3 !== void 0 ? _report$payload$layou3 : 'preserve_layout',
        numPages: (_output$metrics$pdf$p = (_output$metrics$pdf = output.metrics.pdf) === null || _output$metrics$pdf === void 0 ? void 0 : _output$metrics$pdf.pages) !== null && _output$metrics$pdf$p !== void 0 ? _output$metrics$pdf$p : -1,
        screenshotPixels: Math.round((width !== null && width !== void 0 ? width : 0) * (height !== null && height !== void 0 ? height : 0))
      });
    }
    return report;
  }

  // Generic is used to let TS infer the return type at call site.
  async throwIfKibanaShutsDown() {
    await Rx.firstValueFrom(this.opts.reporting.getKibanaShutdown$());
    throw new _reportingCommon.KibanaShuttingDownError();
  }

  /*
   * Provides a TaskRunner for Task Manager
   */
  getTaskRunner() {
    // Keep a separate local stack for each task run
    return ({
      taskInstance,
      fakeRequest
    }) => {
      let jobId;
      const cancellationToken = new _reportingCommon.CancellationToken();
      const {
        retryAt: taskRetryAt,
        startedAt: taskStartedAt
      } = taskInstance;
      return {
        /*
         * Runs a reporting job
         * Claim job: Finds the report in ReportingStore, updates it to "processing"
         * Perform job: Gets the export type's runner, runs it with the job params
         * Complete job: Updates the report in ReportStore with the output from the runner
         * If any error happens, additional retry attempts may be picked up by a separate instance
         */
        run: async () => {
          if (this.kibanaId == null) {
            throw new Error(`Kibana instance ID is undefined!`);
          }
          if (this.kibanaName == null) {
            throw new Error(`Kibana instance name is undefined!`);
          }
          let report;
          const {
            isLastAttempt,
            jobId: jId,
            report: preparedReport,
            task,
            scheduledReport
          } = await this.prepareJob(taskInstance);
          jobId = jId;
          report = preparedReport;
          if (!isLastAttempt) {
            this.opts.reporting.trackReport(jobId);
          }
          if (!report || !task) {
            this.opts.reporting.untrackReport(jobId);
            if (isLastAttempt) {
              (0, _error_logger.errorLogger)(this.logger, `Job ${jobId} failed too many times. Exiting...`);
              return;
            }
            const errorMessage = `Job ${jobId} could not be claimed. Exiting...`;
            (0, _error_logger.errorLogger)(this.logger, errorMessage);

            // Throw so Task manager can clean up the failed task
            throw new Error(`Job ${jobId} could not be claimed. Exiting...`);
          }
          const {
            jobtype: jobType,
            attempts
          } = report;
          const maxAttempts = this.getMaxAttempts();
          if (maxAttempts) {
            this.logger.debug(`Starting ${jobType} report ${jobId}: attempt ${attempts} of ${maxAttempts}.`, {
              tags: [jobId]
            });
          } else {
            this.logger.debug(`Starting ${jobType} report ${jobId}.`, {
              tags: [jobId]
            });
          }
          this.logger.debug(`Reports running: ${this.opts.reporting.countConcurrentReports()}.`, {
            tags: [jobId]
          });
          const eventLog = this.opts.reporting.getEventLogger(new _store.Report({
            ...task,
            _id: task.id,
            _index: task.index
          }));
          try {
            var _output$metrics4;
            const jobContentEncoding = this.getJobContentEncoding(jobType);
            const stream = await (0, _content_stream.getContentStream)(this.opts.reporting, {
              id: report._id,
              index: report._index,
              if_primary_term: report._primary_term,
              if_seq_no: report._seq_no
            }, {
              encoding: jobContentEncoding === 'base64' ? 'base64' : 'raw'
            });
            eventLog.logExecutionStart();
            const output = await Promise.race([this.performJob({
              task,
              fakeRequest,
              taskInstanceFields: {
                retryAt: taskRetryAt,
                startedAt: taskStartedAt
              },
              cancellationToken,
              stream
            }), this.throwIfKibanaShutsDown()]);
            stream.end();
            this.logger.debug(`Begin waiting for the stream's pending callbacks...`, {
              tags: [jobId]
            });
            await (0, _content_stream.finishedWithNoPendingCallbacks)(stream);
            this.logger.info(`The stream's pending callbacks have completed.`, {
              tags: [jobId]
            });
            report._seq_no = stream.getSeqNo();
            report._primary_term = stream.getPrimaryTerm();
            const byteSize = stream.bytesWritten;
            eventLog.logExecutionComplete({
              ...((_output$metrics4 = output.metrics) !== null && _output$metrics4 !== void 0 ? _output$metrics4 : {}),
              byteSize
            });
            if (output) {
              this.logger.debug(`Job output size: ${byteSize} bytes.`, {
                tags: [jobId]
              });
              // Update the job status to "completed"
              report = await this.completeJob(report, {
                ...output,
                size: byteSize
              });
              await this.notify(report, taskInstance, output, byteSize, scheduledReport, task.payload.spaceId);
            }

            // untrack the report for concurrency awareness
            this.logger.debug(`Stopping ${jobId}.`, {
              tags: [jobId]
            });
          } catch (failedToExecuteErr) {
            eventLog.logError(failedToExecuteErr);
            await this.saveExecutionError(report, failedToExecuteErr).catch(failedToSaveError => {
              (0, _error_logger.errorLogger)(this.logger, `Error in saving execution error ${jobId}`, failedToSaveError, [jobId]);
            });
            cancellationToken.cancel();
            const error = (0, _map_to_reporting_error.mapToReportingError)(failedToExecuteErr);
            (0, _server.throwRetryableError)(error, new Date(Date.now() + _.TIME_BETWEEN_ATTEMPTS));
          } finally {
            this.opts.reporting.untrackReport(jobId);
            this.logger.debug(`Reports running: ${this.opts.reporting.countConcurrentReports()}.`, {
              tags: [jobId]
            });
          }
        },
        /*
         * Called by Task Manager to stop the report execution process in case
         * of timeout or server shutdown
         */
        cancel: async () => {
          if (jobId) {
            this.logger.warn(`Cancelling job ${jobId}...`, {
              tags: [jobId]
            });
          }
          cancellationToken.cancel();
        }
      };
    };
  }
}
exports.RunReportTask = RunReportTask;