"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ReportingStore = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _reportingCommon = require("@kbn/reporting-common");
var _reportingServer = require("@kbn/reporting-server");
var _moment = _interopRequireDefault(require("moment"));
var _ = require(".");
var _ilm_policy_manager = require("./ilm_policy_manager");
var _report = require("./report");
var _rollover = require("./rollover");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/*
 * When an instance of Kibana claims a report job, this information tells us about that instance
 */

/*
 * When searching for long-pending reports, we get a subset of fields
 */

/*
 * When searching for long-pending reports, we get a subset of fields
 */
const sourceDoc = doc => {
  return {
    ...doc,
    migration_version: _report.MIGRATION_VERSION,
    '@timestamp': new Date(0).toISOString() // required for data streams compatibility
  };
};
const esDocForUpdate = (report, doc) => {
  return {
    id: report._id,
    index: report._index,
    if_seq_no: report._seq_no,
    if_primary_term: report._primary_term,
    refresh: 'wait_for',
    doc
  };
};
const jobDebugMessage = report => `${report._id} ` + `[_index: ${report._index}] ` + `[_seq_no: ${report._seq_no}]  ` + `[_primary_term: ${report._primary_term}]` + `[attempts: ${report.attempts}] ` + `[process_expiration: ${report.process_expiration}]`;

/*
 * A class to give an interface to historical reports in the reporting.index
 * - track the state: pending, processing, completed, etc
 * - handle updates and deletes to the reporting document
 * - interface for downloading the report
 */
class ReportingStore {
  constructor(reportingCore, logger) {
    (0, _defineProperty2.default)(this, "client", void 0);
    this.reportingCore = reportingCore;
    this.logger = logger;
    this.logger = logger.get('store');
  }
  async getClient() {
    if (!this.client) {
      ({
        asInternalUser: this.client
      } = await this.reportingCore.getEsClient());
    }
    return this.client;
  }
  async createIlmPolicy() {
    const client = await this.getClient();
    const ilmPolicyManager = _ilm_policy_manager.IlmPolicyManager.create({
      client
    });
    if (await ilmPolicyManager.doesIlmPolicyExist()) {
      this.logger.debug(`Found ILM policy ${_reportingCommon.ILM_POLICY_NAME}; skipping creation.`);
    } else {
      this.logger.info(`Creating ILM policy for reporting data stream: ${_reportingCommon.ILM_POLICY_NAME}`);
      await ilmPolicyManager.createIlmPolicy();
    }
    this.logger.info(`Linking ILM policy to reporting data stream: ${_reportingServer.REPORTING_DATA_STREAM_ALIAS}, component template: ${_reportingServer.REPORTING_DATA_STREAM_COMPONENT_TEMPLATE}`);
    await ilmPolicyManager.linkIlmPolicy();
  }
  async indexReport(report) {
    const doc = {
      index: _reportingServer.REPORTING_DATA_STREAM_ALIAS,
      id: report._id,
      refresh: 'wait_for',
      op_type: 'create',
      body: {
        ...report.toReportSource(),
        ...sourceDoc({
          process_expiration: new Date(0).toISOString(),
          attempts: report.attempts || 0,
          status: report.status || _reportingCommon.JOB_STATUS.PENDING
        })
      }
    };
    const client = await this.getClient();
    return await client.index(doc);
  }

  /**
   * Function to be called during plugin start phase. This ensures the environment is correctly
   * configured for storage of reports.
   */
  async start() {
    const {
      statefulSettings
    } = this.reportingCore.getConfig();
    try {
      if (statefulSettings.enabled) {
        await this.createIlmPolicy();
      }
    } catch (e) {
      this.logger.error(`Error creating ILM policy: ${e.message}`, {
        error: {
          stack_trace: e.stack
        }
      });
      throw e;
    }
    try {
      await (0, _rollover.rollDataStreamIfRequired)(this.logger, await this.getClient());
    } catch (e) {
      this.logger.error(`Error rolling over data stream: ${e.message}`, {
        error: {
          stack_trace: e.stack
        }
      });
      // not rethrowing, as this is not a fatal error
    }
  }
  async addReport(report) {
    try {
      report.updateWithEsDoc(await this.indexReport(report));
      return report;
    } catch (err) {
      this.logError(`Error in adding a report!`, err, report);
      throw err;
    }
  }

  /*
   * Search for a report from task data and return back the report
   */
  async findReportFromTask(taskJson) {
    if (!taskJson.index) {
      throw new Error('Task JSON is missing index field!');
    }
    if (!taskJson.id || !taskJson.index) {
      const notRetrievable = new Error(`Unable to retrieve pending report: Invalid report ID!`);
      this.logger.error(notRetrievable); // for stack trace
      throw notRetrievable;
    }
    try {
      var _document$_source, _document$_source2, _document$_source3, _document$_source4, _document$_source5, _document$_source6, _document$_source7, _document$_source8, _document$_source9, _document$_source10, _document$_source11, _document$_source12;
      const client = await this.getClient();
      const document = await client.get({
        index: taskJson.index,
        id: taskJson.id
      });
      return new _.SavedReport({
        _id: document._id,
        _index: document._index,
        _seq_no: document._seq_no,
        _primary_term: document._primary_term,
        jobtype: (_document$_source = document._source) === null || _document$_source === void 0 ? void 0 : _document$_source.jobtype,
        attempts: (_document$_source2 = document._source) === null || _document$_source2 === void 0 ? void 0 : _document$_source2.attempts,
        created_at: (_document$_source3 = document._source) === null || _document$_source3 === void 0 ? void 0 : _document$_source3.created_at,
        created_by: (_document$_source4 = document._source) === null || _document$_source4 === void 0 ? void 0 : _document$_source4.created_by,
        max_attempts: (_document$_source5 = document._source) === null || _document$_source5 === void 0 ? void 0 : _document$_source5.max_attempts,
        meta: (_document$_source6 = document._source) === null || _document$_source6 === void 0 ? void 0 : _document$_source6.meta,
        metrics: (_document$_source7 = document._source) === null || _document$_source7 === void 0 ? void 0 : _document$_source7.metrics,
        payload: (_document$_source8 = document._source) === null || _document$_source8 === void 0 ? void 0 : _document$_source8.payload,
        error: (_document$_source9 = document._source) === null || _document$_source9 === void 0 ? void 0 : _document$_source9.error,
        process_expiration: (_document$_source10 = document._source) === null || _document$_source10 === void 0 ? void 0 : _document$_source10.process_expiration,
        status: (_document$_source11 = document._source) === null || _document$_source11 === void 0 ? void 0 : _document$_source11.status,
        timeout: (_document$_source12 = document._source) === null || _document$_source12 === void 0 ? void 0 : _document$_source12.timeout
      });
    } catch (err) {
      this.logger.error(`Error in finding the report from the scheduled task info! ` + `[id: ${taskJson.id}] [index: ${taskJson.index}]`, {
        error: {
          stack_trace: err.stack
        }
      });
      this.reportingCore.getEventLogger({
        _id: taskJson.id
      }).logError(err);
      throw err;
    }
  }
  async setReportClaimed(report, processingInfo) {
    const doc = sourceDoc({
      ...processingInfo,
      status: _reportingCommon.JOB_STATUS.PROCESSING
    });
    let body;
    try {
      const client = await this.getClient();
      body = await client.update(esDocForUpdate(report, doc));
    } catch (err) {
      this.logError(`Error in updating status to processing! Report: ${jobDebugMessage(report)}`, err, report); // prettier-ignore
      throw err;
    }

    // log the amount of time the report waited in "pending" status
    this.reportingCore.getEventLogger(report).logClaimTask({
      queueDurationMs: _moment.default.utc().valueOf() - _moment.default.utc(report.created_at).valueOf()
    });
    return body;
  }
  logError(message, err, report) {
    this.logger.error(message, {
      error: {
        stack_trace: err.stack
      }
    });
    this.reportingCore.getEventLogger(report).logError(err);
  }
  async setReportFailed(report, failedInfo) {
    const doc = sourceDoc({
      ...failedInfo,
      status: _reportingCommon.JOB_STATUS.FAILED
    });
    let body;
    try {
      const client = await this.getClient();
      body = await client.update(esDocForUpdate(report, doc));
    } catch (err) {
      this.logError(`Error in updating status to failed! Report: ${jobDebugMessage(report)}`, err, report); // prettier-ignore
      throw err;
    }
    this.reportingCore.getEventLogger(report).logReportFailure();
    return body;
  }
  async setReportError(report, errorInfo) {
    const doc = sourceDoc({
      ...errorInfo
    });
    let body;
    try {
      const client = await this.getClient();
      body = await client.update(esDocForUpdate(report, doc));
    } catch (err) {
      this.logError(`Error in updating status to failed! Report: ${jobDebugMessage(report)}`, err, report); // prettier-ignore
      throw err;
    }
    this.reportingCore.getEventLogger(report).logReportFailure();
    return body;
  }
  async setReportCompleted(report, completedInfo) {
    const {
      output
    } = completedInfo;
    const status = output && output.warnings && output.warnings.length > 0 ? _reportingCommon.JOB_STATUS.WARNINGS : _reportingCommon.JOB_STATUS.COMPLETED;
    const doc = sourceDoc({
      ...completedInfo,
      status
    });
    let body;
    try {
      const client = await this.getClient();
      body = await client.update(esDocForUpdate(report, doc));
    } catch (err) {
      this.logError(`Error in updating status to complete! Report: ${jobDebugMessage(report)}`, err, report); // prettier-ignore
      throw err;
    }
    this.reportingCore.getEventLogger(report).logReportSaved();
    return body;
  }
  async setReportWarning(report, warningInfo) {
    var _output$warnings;
    const {
      output,
      warning
    } = warningInfo;
    const warnings = (_output$warnings = output.warnings) !== null && _output$warnings !== void 0 ? _output$warnings : [];
    warnings.push(warning);
    const doc = sourceDoc({
      output: {
        ...output,
        warnings
      },
      status: _reportingCommon.JOB_STATUS.WARNINGS
    });
    let body;
    try {
      const client = await this.getClient();
      body = await client.update(esDocForUpdate(report, doc));
    } catch (err) {
      this.logError(`Error in updating status to warning! Report: ${jobDebugMessage(report)}`, err, report); // prettier-ignore
      throw err;
    }
    return body;
  }
}
exports.ReportingStore = ReportingStore;