"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SavedObjectsSyncService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _configSchema = require("@kbn/config-schema");
var _util = require("./util");
var _service = require("./service");
var _sync = require("./sync");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SAVED_OBJECTS_SYNC_TASK_TYPE = 'ML:saved-objects-sync';
const SAVED_OBJECTS_SYNC_TASK_ID = 'ML:saved-objects-sync-task';
const SAVED_OBJECTS_SYNC_INTERVAL_DEFAULT = '1h';

/**
 * WARNING: Do not modify the existing versioned schema(s) below, instead define a new version (ex: 2, 3, 4).
 * This is required to support zero-downtime upgrades and rollbacks. See https://github.com/elastic/kibana/issues/155764.
 *
 * As you add a new schema version, don't forget to change latestTaskStateSchema variable to reference the latest schema.
 * For example, changing stateSchemaByVersion[1].schema to stateSchemaByVersion[2].schema.
 */
const stateSchemaByVersion = {
  1: {
    // A task that was created < 8.10 will go through this "up" migration
    // to ensure it matches the v1 schema.
    up: state => ({
      runs: state.runs || 0,
      totalSavedObjectsSynced: state.totalSavedObjectsSynced || 0
    }),
    schema: _configSchema.schema.object({
      runs: _configSchema.schema.number(),
      totalSavedObjectsSynced: _configSchema.schema.number()
    })
  }
};
const latestTaskStateSchema = stateSchemaByVersion[1].schema;
class SavedObjectsSyncService {
  constructor(logger) {
    (0, _defineProperty2.default)(this, "core", null);
    (0, _defineProperty2.default)(this, "log", void 0);
    this.log = createLocalLogger(logger, `Task ${SAVED_OBJECTS_SYNC_TASK_ID}: `);
  }
  registerSyncTask(taskManager, security, spacesEnabled, isMlReady) {
    taskManager.registerTaskDefinitions({
      [SAVED_OBJECTS_SYNC_TASK_TYPE]: {
        title: 'ML saved objet sync',
        description: "This task periodically syncs ML's saved objects",
        timeout: '1m',
        maxAttempts: 3,
        stateSchemaByVersion,
        createTaskRunner: ({
          taskInstance
        }) => {
          return {
            run: async () => {
              await isMlReady();
              const core = this.core;
              const state = taskInstance.state;
              if (core === null || security === null || spacesEnabled === null) {
                const error = 'dependencies not initialized';
                this.log.error(error);
                throw new Error(error);
              }
              const client = core.elasticsearch.client;
              const {
                getInternalSavedObjectsClient
              } = (0, _util.savedObjectClientsFactory)(() => core.savedObjects);
              const savedObjectsClient = getInternalSavedObjectsClient();
              if (savedObjectsClient === null) {
                const error = 'Internal saved object client not initialized';
                this.log.error(error);
                throw new Error(error);
              }
              const mlSavedObjectService = (0, _service.mlSavedObjectServiceFactory)(savedObjectsClient, savedObjectsClient, spacesEnabled, security === null || security === void 0 ? void 0 : security.authz, client, isMlReady);
              const {
                initSavedObjects
              } = (0, _sync.syncSavedObjectsFactory)(client, mlSavedObjectService);
              const {
                jobs,
                trainedModels
              } = await initSavedObjects(false);
              const count = jobs.length + trainedModels.length;
              this.log.info(count ? `${count} ML saved object${count > 1 ? 's' : ''} synced` : 'No ML saved objects in need of synchronization');
              const updatedState = {
                runs: state.runs + 1,
                totalSavedObjectsSynced: state.totalSavedObjectsSynced + count
              };
              return {
                state: updatedState
              };
            },
            cancel: async () => {
              this.log.warn('timed out');
            }
          };
        }
      }
    });
  }
  async scheduleSyncTask(taskManager, core) {
    this.core = core;
    try {
      var _taskInstance$schedul;
      const state = {
        runs: 0,
        totalSavedObjectsSynced: 0
      };
      const taskInstance = await taskManager.ensureScheduled({
        id: SAVED_OBJECTS_SYNC_TASK_ID,
        taskType: SAVED_OBJECTS_SYNC_TASK_TYPE,
        schedule: {
          interval: SAVED_OBJECTS_SYNC_INTERVAL_DEFAULT
        },
        params: {},
        state,
        scope: ['ml']
      });
      this.log.info(`scheduled with interval ${(_taskInstance$schedul = taskInstance.schedule) === null || _taskInstance$schedul === void 0 ? void 0 : _taskInstance$schedul.interval}`);
      return taskInstance;
    } catch (e) {
      var _e$message;
      this.log.error(`Error running task: ${SAVED_OBJECTS_SYNC_TASK_ID}, `, (_e$message = e === null || e === void 0 ? void 0 : e.message) !== null && _e$message !== void 0 ? _e$message : e);
      return null;
    }
  }
  async unscheduleSyncTask(taskManager) {
    await taskManager.removeIfExists(SAVED_OBJECTS_SYNC_TASK_ID);
  }
}
exports.SavedObjectsSyncService = SavedObjectsSyncService;
function createLocalLogger(logger, preText) {
  return {
    info: text => logger.info(`${preText}${text}`),
    warn: text => logger.warn(`${preText}${text}`),
    error: (text, e) => logger.error(`${preText}${text} ${e !== null && e !== void 0 ? e : ''}`)
  };
}