"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getColorSpec = exports.USER_SELECTION = exports.SINGLE_POINT_CLICK = exports.OUTLIER_SCORE_FIELD = exports.COLOR_SELECTION = exports.COLOR_RANGE_QUANTITATIVE = exports.COLOR_RANGE_OUTLIER = exports.COLOR_RANGE_NOMINAL = exports.COLOR_OUTLIER = exports.COLOR_BLUR = void 0;
exports.getEscapedVegaFieldName = getEscapedVegaFieldName;
exports.getScatterplotMatrixVegaLiteSpec = void 0;
var _eui = require("@elastic/eui");
var _i18n = require("@kbn/i18n");
var _common = require("../vega_chart/common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// There is still an issue with Vega Lite's typings with the strict mode Kibana is using.
// @ts-ignore

const OUTLIER_SCORE_FIELD = exports.OUTLIER_SCORE_FIELD = 'outlier_score';
const SCATTERPLOT_SIZE = 125;
const USER_SELECTION = exports.USER_SELECTION = 'user_selection';
const SINGLE_POINT_CLICK = exports.SINGLE_POINT_CLICK = 'single_point_click';
const COLOR_BLUR = exports.COLOR_BLUR = '#bbb';
const COLOR_OUTLIER = exports.COLOR_OUTLIER = (0, _eui.euiPaletteRed)(2)[1];
const COLOR_SELECTION = exports.COLOR_SELECTION = (0, _eui.euiPaletteColorBlind)()[2];
const COLOR_RANGE_OUTLIER = exports.COLOR_RANGE_OUTLIER = [(0, _eui.euiPaletteColorBlind)()[1], (0, _eui.euiPaletteColorBlind)()[2]];
const COLOR_RANGE_NOMINAL = exports.COLOR_RANGE_NOMINAL = (0, _eui.euiPaletteColorBlind)({
  rotations: 2
});
const COLOR_RANGE_QUANTITATIVE = exports.COLOR_RANGE_QUANTITATIVE = (0, _eui.euiPaletteGreen)(5);
const CUSTOM_VIS_FIELDS_PATH = 'fields';
const getColorSpec = (forCustomVisLink, euiTheme, escapedOutlierScoreField, color, legendType) => {
  // For outlier detection result pages coloring is done based on a threshold.
  // This returns a Vega spec using a conditional to return the color.
  if (typeof escapedOutlierScoreField === 'string') {
    var _is_outlier;
    return {
      condition: {
        selection: USER_SELECTION,
        // @ts-expect-error upgrade typescript v5.9.3
        field: getEscapedVegaFieldName((_is_outlier = 'is_outlier') !== null && _is_outlier !== void 0 ? _is_outlier : '00FF00'),
        type: _common.LEGEND_TYPES.NOMINAL,
        scale: {
          range: COLOR_RANGE_OUTLIER
        }
      },
      value: COLOR_BLUR
    };
  }

  // Based on the type of the color field,
  // this returns either a continuous or categorical color spec.
  if (color !== undefined && legendType !== undefined) {
    return {
      condition: {
        selection: USER_SELECTION,
        field: `${forCustomVisLink ? `${CUSTOM_VIS_FIELDS_PATH}.` : ''}${getEscapedVegaFieldName(color !== null && color !== void 0 ? color : '00FF00' // When creating the custom link - this field is returned in an array so we need to access it
        )}${forCustomVisLink ? '[0]' : ''}`,
        type: legendType,
        scale: {
          range: legendType === _common.LEGEND_TYPES.NOMINAL ? COLOR_RANGE_NOMINAL : COLOR_RANGE_QUANTITATIVE
        }
      },
      value: COLOR_BLUR
    };
  }
  return {
    condition: [{
      selection: USER_SELECTION
    }, {
      selection: SINGLE_POINT_CLICK
    }],
    value: COLOR_BLUR
  };
};
exports.getColorSpec = getColorSpec;
const getVegaSpecLayer = (forCustomVisLink, isBackground, values, colorSpec, escapedOutlierScoreField, outliers, dynamicSize, vegaColumns, color) => {
  const selection = outliers ? {
    selection: {
      [USER_SELECTION]: {
        type: 'interval'
      },
      [SINGLE_POINT_CLICK]: {
        type: 'single'
      },
      mlOutlierScoreThreshold: {
        type: 'single',
        fields: ['cutoff'],
        bind: {
          input: 'range',
          max: 1,
          min: 0,
          name: _i18n.i18n.translate('xpack.ml.splomSpec.outlierScoreThresholdName', {
            defaultMessage: 'Outlier score threshold: '
          }),
          step: 0.01
        },
        init: {
          cutoff: 0.99
        }
      }
    }
  } : {
    selection: {
      // Always allow user selection
      [USER_SELECTION]: {
        type: 'interval'
      },
      [SINGLE_POINT_CLICK]: {
        type: 'single',
        empty: 'none'
      }
    }
  };
  return {
    // Don't need to add static data for custom vis links
    ...(forCustomVisLink ? {} : {
      data: {
        values: [...values]
      }
    }),
    mark: {
      ...(outliers && dynamicSize ? {
        type: 'circle',
        strokeWidth: 1.2,
        strokeOpacity: 0.75,
        fillOpacity: 0.1
      } : {
        type: 'circle',
        opacity: 0.75,
        size: 8
      })
    },
    // transformation to apply outlier threshold as category
    ...(outliers ? {
      transform: [{
        calculate: `datum${forCustomVisLink ? `.${CUSTOM_VIS_FIELDS_PATH}` : ''}['${escapedOutlierScoreField}'] >= mlOutlierScoreThreshold.cutoff`,
        as: 'is_outlier'
      }]
    } : {}),
    encoding: {
      color: isBackground ? {
        value: COLOR_BLUR
      } : colorSpec,
      opacity: {
        condition: {
          selection: USER_SELECTION,
          value: 0.8
        },
        value: 0.5
      },
      ...(dynamicSize ? {
        stroke: colorSpec,
        opacity: {
          condition: {
            value: 1,
            test: `(datum${forCustomVisLink ? `.${CUSTOM_VIS_FIELDS_PATH}` : ''}['${escapedOutlierScoreField}'] >= mlOutlierScoreThreshold.cutoff)`
          },
          value: 0.5
        }
      } : {}),
      ...(outliers ? {
        order: {
          field: `${forCustomVisLink ? `${CUSTOM_VIS_FIELDS_PATH}.` : ''}${escapedOutlierScoreField}`
        },
        size: {
          ...(!dynamicSize ? {
            condition: {
              value: 40,
              test: `(datum${forCustomVisLink ? `.${CUSTOM_VIS_FIELDS_PATH}` : ''}['${escapedOutlierScoreField}'] >= mlOutlierScoreThreshold.cutoff)`
            },
            value: 8
          } : {
            type: _common.LEGEND_TYPES.QUANTITATIVE,
            field: `${forCustomVisLink ? `${CUSTOM_VIS_FIELDS_PATH}.` : ''}${escapedOutlierScoreField}`,
            scale: {
              type: 'linear',
              range: [8, 200],
              domain: [0, 1]
            }
          })
        }
      } : {}),
      x: {
        type: _common.LEGEND_TYPES.QUANTITATIVE,
        field: {
          repeat: 'column'
        },
        scale: {
          zero: false
        }
      },
      y: {
        type: _common.LEGEND_TYPES.QUANTITATIVE,
        field: {
          repeat: 'row'
        },
        scale: {
          zero: false
        }
      },
      tooltip: [...(color !== undefined ?
      // @ts-ignore
      [{
        type: colorSpec.condition.type,
        field: `${forCustomVisLink ? `${CUSTOM_VIS_FIELDS_PATH}.` : ''}${getEscapedVegaFieldName(color)}`
      }] : []), ...vegaColumns.map(d => ({
        type: _common.LEGEND_TYPES.QUANTITATIVE,
        field: d
      })), ...(outliers ? [{
        type: _common.LEGEND_TYPES.QUANTITATIVE,
        field: `${forCustomVisLink ? `${CUSTOM_VIS_FIELDS_PATH}.` : ''}${escapedOutlierScoreField}`,
        format: '.3f'
      }] : [])]
    },
    ...(isBackground ? {} : selection),
    ...(forCustomVisLink ? {} : {
      width: SCATTERPLOT_SIZE
    }),
    ...(forCustomVisLink ? {} : {
      height: SCATTERPLOT_SIZE
    })
  };
};

// Escapes the characters .[]\ in field names with double backslashes
// since VEGA treats dots/brackets in field names as nested values.
// See https://vega.github.io/vega-lite/docs/field.html for details.
function getEscapedVegaFieldName(fieldName, prependString = '') {
  // Note the following isn't 100% ideal because there are cases when we may
  // end up with an additional backslash being rendered for labels of the
  // scatterplot. However, all other variations I tried caused rendering
  // problems of the charts and rendering would fail completely.

  // For example, just escaping \n in the first replace without the general
  // backslash escaping causes the following Vega error:
  // Duplicate scale or projection name: "child__row_my_numbercolumn_my_number_x"

  // Escaping just the backslash without the additional \n escaping causes
  // causes an "expression parse error" in Vega and the chart wouldn't render.

  // Escape newline characters
  fieldName = fieldName.replace(/\n/g, '\\n');
  // Escape .[]\
  fieldName = fieldName.replace(/([\.|\[|\]|\\])/g, '\\$1');
  return `${prependString}${fieldName}`;
}
const getScatterplotMatrixVegaLiteSpec = (forCustomVisLink, values, backgroundValues, columns, euiTheme, resultsField, color, legendType, dynamicSize) => {
  const vegaValues = values;
  const vegaColumns = columns.map(column => getEscapedVegaFieldName(column, forCustomVisLink ? 'fields.' : ''));
  const outliers = resultsField !== undefined;
  const escapedOutlierScoreField = `${resultsField}\\.${OUTLIER_SCORE_FIELD}`;
  const colorSpec = getColorSpec(forCustomVisLink, euiTheme, resultsField && escapedOutlierScoreField, color, legendType);
  const schema = {
    $schema: 'https://vega.github.io/schema/vega-lite/v4.17.0.json',
    background: 'transparent',
    // There seems to be a bug in Vega which doesn't propagate these settings
    // for repeated charts, it seems to be fixed for facets but not repeat.
    // This causes #ddd lines to stand out in dark mode.
    // See: https://github.com/vega/vega-lite/issues/5908
    view: {
      fill: 'transparent',
      stroke: euiTheme.euiColorLightestShade
    },
    padding: 10,
    config: {
      axis: {
        domainColor: euiTheme.euiColorLightShade,
        gridColor: euiTheme.euiColorLightestShade,
        tickColor: euiTheme.euiColorLightestShade,
        labelColor: euiTheme.euiTextSubduedColor,
        titleColor: euiTheme.euiTextSubduedColor
      },
      legend: {
        orient: 'right',
        labelColor: euiTheme.euiTextSubduedColor,
        titleColor: euiTheme.euiTextSubduedColor
      }
    },
    repeat: {
      column: vegaColumns,
      row: vegaColumns.slice().reverse()
    },
    spec: {
      layer: [getVegaSpecLayer(forCustomVisLink, false, vegaValues, colorSpec, escapedOutlierScoreField, outliers, !!dynamicSize, vegaColumns, color)]
    }
  };
  if (backgroundValues.length) {
    schema.spec.layer.unshift(getVegaSpecLayer(forCustomVisLink, true, backgroundValues, colorSpec, escapedOutlierScoreField, outliers, !!dynamicSize, vegaColumns, color));
  }
  return schema;
};
exports.getScatterplotMatrixVegaLiteSpec = getScatterplotMatrixVegaLiteSpec;