"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAvailableOperationsByMetadata = getAvailableOperationsByMetadata;
exports.getCurrentFieldsForOperation = getCurrentFieldsForOperation;
exports.getOperationDisplay = void 0;
exports.getOperationHelperForMultipleFields = getOperationHelperForMultipleFields;
exports.getOperationTypesForField = getOperationTypesForField;
exports.getOperations = getOperations;
exports.getSortScoreByPriority = getSortScoreByPriority;
exports.getSortScoreByPriorityForField = void 0;
exports.hasOperationSupportForMultipleFields = hasOperationSupportForMultipleFields;
exports.isDocumentOperation = isDocumentOperation;
exports.memoizedGetAvailableOperationsByMetadata = void 0;
Object.defineProperty(exports, "operationDefinitionMap", {
  enumerable: true,
  get: function () {
    return _definitions.operationDefinitionMap;
  }
});
exports.translateToOperationName = translateToOperationName;
var _lodash = require("lodash");
var _definitions = require("./definitions");
var _document_field = require("../document_field");
var _pure_utils = require("../pure_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Map aggregation names from Elasticsearch to Lens names.
 * Used when loading indexpatterns to map metadata (i.e. restrictions)
 */
function translateToOperationName(agg) {
  return agg in _definitions.renameOperationsMapping ? _definitions.renameOperationsMapping[agg] : agg;
}
/**
 * Returns all available operation types as a list at runtime.
 * This will be an array of each member of the union type `OperationType`
 * without any guaranteed order
 */
function getOperations() {
  return Object.keys(_definitions.operationDefinitionMap);
}

/**
 * Returns a list of the display names of all operations with any guaranteed order.
 */
const getOperationDisplay = exports.getOperationDisplay = (0, _lodash.memoize)(() => {
  const display = {};
  _definitions.operationDefinitions.forEach(({
    type,
    displayName
  }) => {
    display[type] = {
      type,
      displayName
    };
  });
  return display;
});
function getSortScoreByPriority(a, b) {
  return (b.priority || Number.NEGATIVE_INFINITY) - (a.priority || Number.NEGATIVE_INFINITY);
}
const getSortScoreByPriorityForField = field => (a, b) => {
  var _field$partiallyAppli, _field$partiallyAppli2;
  if ((field === null || field === void 0 ? void 0 : (_field$partiallyAppli = field.partiallyApplicableFunctions) === null || _field$partiallyAppli === void 0 ? void 0 : _field$partiallyAppli[a.type]) !== (field === null || field === void 0 ? void 0 : (_field$partiallyAppli2 = field.partiallyApplicableFunctions) === null || _field$partiallyAppli2 === void 0 ? void 0 : _field$partiallyAppli2[b.type])) {
    var _field$partiallyAppli3;
    if (field !== null && field !== void 0 && (_field$partiallyAppli3 = field.partiallyApplicableFunctions) !== null && _field$partiallyAppli3 !== void 0 && _field$partiallyAppli3[a.type]) return 1;
    return -1;
  }
  return (b.priority || Number.NEGATIVE_INFINITY) - (a.priority || Number.NEGATIVE_INFINITY);
};
exports.getSortScoreByPriorityForField = getSortScoreByPriorityForField;
function getCurrentFieldsForOperation(targetColumn) {
  var _operationDefinitionM, _operationDefinitionM2, _operationDefinitionM3;
  if (!(0, _pure_utils.hasField)(targetColumn)) {
    return [];
  }
  return (_operationDefinitionM = (_operationDefinitionM2 = _definitions.operationDefinitionMap[targetColumn.operationType]) === null || _operationDefinitionM2 === void 0 ? void 0 : (_operationDefinitionM3 = _operationDefinitionM2.getCurrentFields) === null || _operationDefinitionM3 === void 0 ? void 0 : _operationDefinitionM3.call(_operationDefinitionM2, targetColumn)) !== null && _operationDefinitionM !== void 0 ? _operationDefinitionM : [targetColumn.sourceField];
}
function getOperationHelperForMultipleFields(operationType) {
  var _operationDefinitionM4;
  return (_operationDefinitionM4 = _definitions.operationDefinitionMap[operationType]) === null || _operationDefinitionM4 === void 0 ? void 0 : _operationDefinitionM4.getParamsForMultipleFields;
}
function hasOperationSupportForMultipleFields(indexPattern, targetColumn, sourceColumn, field) {
  var _operationDefinitionM5, _operationDefinitionM6;
  return Boolean((_operationDefinitionM5 = _definitions.operationDefinitionMap[targetColumn.operationType]) === null || _operationDefinitionM5 === void 0 ? void 0 : (_operationDefinitionM6 = _operationDefinitionM5.canAddNewField) === null || _operationDefinitionM6 === void 0 ? void 0 : _operationDefinitionM6.call(_operationDefinitionM5, {
    targetColumn,
    sourceColumn,
    field,
    indexPattern
  }));
}

/**
 * Returns all `OperationType`s that can build a column using `buildColumn` based on the
 * passed in field.
 */
function getOperationTypesForField(field, filterOperations, alreadyUsedOperations) {
  return _definitions.operationDefinitions.filter(operationDefinition => {
    if (operationDefinition.input !== 'field') {
      return false;
    }
    const possibleOperation = operationDefinition.getPossibleOperationForField(field);
    return filterOperations ? possibleOperation && filterOperations(possibleOperation) : possibleOperation;
  }).sort(getSortScoreByPriorityForField(field)).sort((a, b) => {
    if (!alreadyUsedOperations) return 0;
    // if some operations are used already, order them so the unused operations come first
    const aAlreadyUsed = alreadyUsedOperations.has(a.type);
    const bAlreadyUsed = alreadyUsedOperations.has(b.type);
    if (aAlreadyUsed === bAlreadyUsed) return 0;
    if (aAlreadyUsed) return 1;
    return -1;
  }).map(({
    type
  }) => type);
}
let documentOperations;
function isDocumentOperation(type) {
  // This can't be done at the root level, because it breaks tests, thanks to mocking oddities
  // so we do it here, and cache the result.
  documentOperations = documentOperations || new Set(getOperationTypesForField(_document_field.documentField));
  return documentOperations.has(type);
}
/**
 * Returns all possible operations (matches between operations and fields of the index
 * pattern plus matches for operations and documents of the index pattern) indexed by the
 * meta data of the operation.
 *
 * The resulting list is filtered down by the `filterOperations` function passed in by
 * the current visualization to determine which operations and field are applicable for
 * a given dimension.
 *
 * Example output:
 * ```
 * [
 *    {
 *      operationMetaData: { dataType: 'string', isBucketed: true },
 *      operations: [{
 *        type: 'field',
 *        operationType: ['terms'],
 *        field: 'keyword'
 *      }]
 *    },
 *    {
 *      operationMetaData: { dataType: 'string', isBucketed: true },
 *      operations: [{
 *        type: 'none',
 *        operationType: ['filters'],
 *      }]
 *    },
 * ]
 * ```
 */
function getAvailableOperationsByMetadata(indexPattern,
// For consistency in testing
customOperationDefinitionMap) {
  const operationByMetadata = {};
  const addToMap = (operation, operationMetadata) => {
    if (!operationMetadata) return;
    const key = JSON.stringify(operationMetadata);
    if (operationByMetadata[key]) {
      operationByMetadata[key].operations.push(operation);
    } else {
      operationByMetadata[key] = {
        operationMetaData: operationMetadata,
        operations: [operation]
      };
    }
  };
  (customOperationDefinitionMap ? Object.values(customOperationDefinitionMap) : _definitions.operationDefinitions).sort(getSortScoreByPriority).forEach(operationDefinition => {
    if (operationDefinition.input === 'field') {
      indexPattern.fields.forEach(field => {
        addToMap({
          type: 'field',
          operationType: operationDefinition.type,
          field: field.name
        }, operationDefinition.getPossibleOperationForField(field));
      });
    } else if (operationDefinition.input === 'none') {
      const validOperation = operationDefinition.getPossibleOperation(indexPattern);
      if (validOperation) {
        addToMap({
          type: 'none',
          operationType: operationDefinition.type
        }, validOperation);
      }
    } else if (operationDefinition.input === 'fullReference') {
      const validOperation = operationDefinition.getPossibleOperation(indexPattern);
      if (validOperation) {
        addToMap({
          type: 'fullReference',
          operationType: operationDefinition.type
        }, validOperation);
      }
    } else if (operationDefinition.input === 'managedReference') {
      const validOperation = operationDefinition.getPossibleOperation();
      if (validOperation) {
        addToMap({
          type: 'managedReference',
          operationType: operationDefinition.type,
          usedInMath: operationDefinition.usedInMath
        }, validOperation);
      }
    }
  });
  return Object.values(operationByMetadata);
}
const memoizedGetAvailableOperationsByMetadata = exports.memoizedGetAvailableOperationsByMetadata = (0, _lodash.memoize)(getAvailableOperationsByMetadata);