"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerAggregateRoute = exports.doSearch = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _configSchema = require("@kbn/config-schema");
var _constants = require("../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// sort by values
const ASC = 'asc';
const DESC = 'desc';
const registerAggregateRoute = (router, logger) => {
  router.versioned.get({
    access: 'internal',
    path: _constants.AGGREGATE_ROUTE
  }).addVersion({
    version: '1',
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    validate: {
      request: {
        query: _configSchema.schema.object({
          index: _configSchema.schema.string(),
          query: _configSchema.schema.string(),
          countBy: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal(_constants.ORCHESTRATOR_CLUSTER_ID), _configSchema.schema.literal(_constants.ORCHESTRATOR_RESOURCE_ID), _configSchema.schema.literal(_constants.ORCHESTRATOR_NAMESPACE), _configSchema.schema.literal(_constants.ORCHESTRATOR_CLUSTER_NAME), _configSchema.schema.literal(_constants.CLOUD_INSTANCE_NAME), _configSchema.schema.literal(_constants.CONTAINER_IMAGE_NAME), _configSchema.schema.literal(_constants.ENTRY_LEADER_ENTITY_ID)])),
          groupBy: _configSchema.schema.oneOf([_configSchema.schema.literal(_constants.ORCHESTRATOR_CLUSTER_ID), _configSchema.schema.literal(_constants.ORCHESTRATOR_RESOURCE_ID), _configSchema.schema.literal(_constants.ORCHESTRATOR_NAMESPACE), _configSchema.schema.literal(_constants.ORCHESTRATOR_CLUSTER_NAME), _configSchema.schema.literal(_constants.CLOUD_INSTANCE_NAME), _configSchema.schema.literal(_constants.CONTAINER_IMAGE_NAME), _configSchema.schema.literal(_constants.ENTRY_LEADER_USER_ID), _configSchema.schema.literal(_constants.ENTRY_LEADER_INTERACTIVE)]),
          page: _configSchema.schema.number({
            defaultValue: 0,
            max: 10000,
            min: 0
          }),
          perPage: _configSchema.schema.maybe(_configSchema.schema.number({
            defaultValue: 10,
            max: 100,
            min: 1
          })),
          sortByCount: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal(ASC), _configSchema.schema.literal(DESC)]))
        })
      }
    }
  }, async (context, request, response) => {
    const client = (await context.core).elasticsearch.client.asCurrentUser;
    const {
      query,
      countBy,
      sortByCount,
      groupBy,
      page,
      perPage,
      index
    } = request.query;
    try {
      const body = await doSearch(client, index, query, groupBy, page, perPage, countBy, sortByCount);
      return response.ok({
        body
      });
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      logger.error(`Failed to fetch k8s aggregates: ${err}`);
      return response.customError({
        body: {
          message: error.message
        },
        statusCode: error.statusCode
      });
    }
  });
};
exports.registerAggregateRoute = registerAggregateRoute;
const doSearch = async (client, index, query, groupBy, page,
// zero based
perPage = _constants.AGGREGATE_PAGE_SIZE, countBy, sortByCount) => {
  var _search$aggregations;
  const queryDSL = JSON.parse(query);
  const countByAggs = countBy ? {
    count_by_aggs: {
      cardinality: {
        field: countBy
      }
    }
  } : undefined;
  let sort = {
    _key: {
      order: ASC
    }
  };
  if (sortByCount === ASC || sortByCount === DESC) {
    sort = {
      'count_by_aggs.value': {
        order: sortByCount
      }
    };
  }
  const search = await client.search({
    index: [index],
    body: {
      query: queryDSL,
      size: 0,
      aggs: {
        custom_agg: {
          terms: {
            field: groupBy,
            size: _constants.AGGREGATE_MAX_BUCKETS
          },
          aggs: {
            ...countByAggs,
            bucket_sort: {
              bucket_sort: {
                sort: [sort],
                // defaulting to alphabetic sort
                size: perPage + 1,
                // check if there's a "next page"
                from: perPage * page
              }
            }
          }
        }
      }
    }
  });
  const agg = (_search$aggregations = search.aggregations) === null || _search$aggregations === void 0 ? void 0 : _search$aggregations.custom_agg;
  const buckets = (agg === null || agg === void 0 ? void 0 : agg.buckets) || [];
  const hasNextPage = buckets.length > perPage;
  if (hasNextPage) {
    buckets.pop();
  }
  return {
    buckets,
    hasNextPage
  };
};
exports.doSearch = doSearch;