"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.InfraServerPlugin = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/es-ui-shared-plugin/server");
var _i18n = require("@kbn/i18n");
var _common = require("@kbn/spaces-plugin/common");
var _common2 = require("@kbn/observability-shared-plugin/common");
var _common3 = require("@kbn/observability-plugin/common");
var _lodash = require("lodash");
var _constants = require("../common/constants");
var _features = require("./features");
var _infra_server = require("./infra_server");
var _kibana_framework_adapter = require("./lib/adapters/framework/kibana_framework_adapter");
var _kibana_metrics_adapter = require("./lib/adapters/metrics/kibana_metrics_adapter");
var _source_status = require("./lib/adapters/source_status");
var _alerting = require("./lib/alerting");
var _register_rule_types = require("./lib/alerting/register_rule_types");
var _metrics_domain = require("./lib/domains/metrics_domain");
var _sources = require("./lib/sources");
var _source_status2 = require("./lib/source_status");
var _saved_objects = require("./saved_objects");
var _inventory_views = require("./services/inventory_views");
var _metrics_explorer_views = require("./services/metrics_explorer_views");
var _rules = require("./services/rules");
var _usage_collector = require("./usage/usage_collector");
var _map_source_to_log_view = require("./utils/map_source_to_log_view");
var _ui_settings = require("../common/ui_settings");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const logsSampleDataLinkLabel = _i18n.i18n.translate('xpack.infra.sampleDataLinkLabel', {
  defaultMessage: 'Logs'
});
class InfraServerPlugin {
  constructor(context) {
    (0, _defineProperty2.default)(this, "config", void 0);
    (0, _defineProperty2.default)(this, "libs", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "logsRules", void 0);
    (0, _defineProperty2.default)(this, "metricsRules", void 0);
    (0, _defineProperty2.default)(this, "inventoryViews", void 0);
    (0, _defineProperty2.default)(this, "metricsExplorerViews", void 0);
    this.config = context.config.get();
    this.logger = context.logger.get();
    this.logsRules = new _rules.RulesService(_constants.LOGS_FEATURE_ID, _register_rule_types.LOGS_RULES_ALERT_CONTEXT, this.logger.get('logsRules'));
    this.metricsRules = new _rules.RulesService(_constants.METRICS_FEATURE_ID, _register_rule_types.METRICS_RULES_ALERT_CONTEXT, this.logger.get('metricsRules'));
    this.inventoryViews = new _inventory_views.InventoryViewsService(this.logger.get('inventoryViews'));
    this.metricsExplorerViews = this.config.featureFlags.metricsExplorerEnabled ? new _metrics_explorer_views.MetricsExplorerViewsService(this.logger.get('metricsExplorerViews')) : undefined;
  }
  setup(core, plugins) {
    var _this$metricsExplorer, _this$config$sources, _this$config$sources$, _this$config$sources$2;
    const framework = new _kibana_framework_adapter.KibanaFramework(core, this.config, plugins);
    const metricsClient = plugins.metricsDataAccess.client;
    metricsClient.setDefaultMetricIndicesHandler(async options => {
      const sourceConfiguration = await sources.getInfraSourceConfiguration(options.savedObjectsClient, 'default');
      return sourceConfiguration.configuration.metricAlias;
    });
    const sources = new _sources.InfraSources({
      metricsClient
    });
    const sourceStatus = new _source_status2.InfraSourceStatus(new _source_status.InfraElasticsearchSourceStatusAdapter(framework), {
      sources
    });
    const alertsLocator = plugins.share.url.locators.get(_common3.alertsLocatorID);
    const assetDetailsLocator = plugins.share.url.locators.create(new _common2.AssetDetailsLocatorDefinition());
    const metricsExplorerLocator = plugins.share.url.locators.create(new _common2.MetricsExplorerLocatorDefinition());
    const inventoryLocator = plugins.share.url.locators.create(new _common2.InventoryLocatorDefinition());

    // Setup infra services
    const inventoryViews = this.inventoryViews.setup();
    const metricsExplorerViews = (_this$metricsExplorer = this.metricsExplorerViews) === null || _this$metricsExplorer === void 0 ? void 0 : _this$metricsExplorer.setup();

    // Register uiSettings config
    core.uiSettings.register(_ui_settings.uiSettings);

    // Register saved object types
    core.savedObjects.registerType(_sources.infraSourceConfigurationSavedObjectType);
    core.savedObjects.registerType(_saved_objects.inventoryViewSavedObjectType);
    core.savedObjects.registerType(_saved_objects.infraCustomDashboardsSavedObjectType);
    if (this.config.featureFlags.metricsExplorerEnabled) {
      core.savedObjects.registerType(_saved_objects.metricsExplorerViewSavedObjectType);
    }

    // TODO: separate these out individually and do away with "domains" as a temporary group
    // and make them available via the request context so we can do away with
    // the wrapper classes
    const domainLibs = {
      logEntries: plugins.logsShared.logEntries,
      metrics: new _metrics_domain.InfraMetricsDomain(new _kibana_metrics_adapter.KibanaMetricsAdapter(framework))
    };

    // Instead of passing plugins individually to `libs` on a necessity basis,
    // this provides an object with all plugins infra depends on
    const libsPlugins = (0, _lodash.mapValues)(plugins, (value, key) => {
      return {
        setup: value,
        start: () => core.getStartServices().then(services => {
          const [, pluginsStartContracts] = services;
          return pluginsStartContracts[key];
        })
      };
    });
    this.libs = {
      configuration: this.config,
      framework,
      sources,
      sourceStatus,
      ...domainLibs,
      handleEsError: _server.handleEsError,
      logsRules: this.logsRules.setup(core, plugins),
      metricsRules: this.metricsRules.setup(core, plugins),
      getStartServices: () => core.getStartServices(),
      getAlertDetailsConfig: () => plugins.observability.getAlertDetailsConfig(),
      logger: this.logger,
      basePath: core.http.basePath,
      plugins: libsPlugins
    };
    plugins.features.registerKibanaFeature((0, _features.getMetricsFeature)());
    plugins.features.registerKibanaFeature((0, _features.getLogsFeature)());

    // Register an handler to retrieve the fallback logView starting from a source configuration
    plugins.logsShared.logViews.registerLogViewFallbackHandler(async (sourceId, {
      soClient
    }) => {
      const sourceConfiguration = await sources.getInfraSourceConfiguration(soClient, sourceId);
      return (0, _map_source_to_log_view.mapSourceToLogView)(sourceConfiguration);
    });
    plugins.logsShared.logViews.setLogViewsStaticConfig({
      messageFields: (_this$config$sources = this.config.sources) === null || _this$config$sources === void 0 ? void 0 : (_this$config$sources$ = _this$config$sources.default) === null || _this$config$sources$ === void 0 ? void 0 : (_this$config$sources$2 = _this$config$sources$.fields) === null || _this$config$sources$2 === void 0 ? void 0 : _this$config$sources$2.message
    });
    plugins.logsShared.registerUsageCollectorActions({
      countLogs: () => _usage_collector.UsageCollector.countLogs()
    });
    if (this.config.featureFlags.logsUIEnabled) {
      plugins.home.sampleData.addAppLinksToSampleDataset('logs', [{
        sampleObject: null,
        // indicates that there is no sample object associated with this app link's path
        getPath: () => `/app/logs`,
        label: logsSampleDataLinkLabel,
        icon: 'logsApp'
      }]);
    }
    (0, _alerting.registerRuleTypes)(plugins.alerting, this.libs, this.config, {
      alertsLocator,
      assetDetailsLocator,
      metricsExplorerLocator,
      inventoryLocator
    });
    core.http.registerRouteHandlerContext('infra', async (context, request) => {
      var _plugins$ml, _plugins$ml2, _plugins$spaces$space, _plugins$spaces;
      const coreContext = await context.core;
      const savedObjectsClient = coreContext.savedObjects.client;
      const uiSettingsClient = coreContext.uiSettings.client;
      const mlSystem = (_plugins$ml = plugins.ml) === null || _plugins$ml === void 0 ? void 0 : _plugins$ml.mlSystemProvider(request, savedObjectsClient);
      const mlAnomalyDetectors = (_plugins$ml2 = plugins.ml) === null || _plugins$ml2 === void 0 ? void 0 : _plugins$ml2.anomalyDetectorsProvider(request, savedObjectsClient);
      const spaceId = (_plugins$spaces$space = (_plugins$spaces = plugins.spaces) === null || _plugins$spaces === void 0 ? void 0 : _plugins$spaces.spacesService.getSpaceId(request)) !== null && _plugins$spaces$space !== void 0 ? _plugins$spaces$space : _common.DEFAULT_SPACE_ID;
      const getMetricsIndices = async () => {
        return metricsClient.getMetricIndices({
          savedObjectsClient
        });
      };
      return {
        mlAnomalyDetectors,
        mlSystem,
        spaceId,
        savedObjectsClient,
        uiSettingsClient,
        getMetricsIndices
      };
    });

    // Telemetry
    _usage_collector.UsageCollector.registerUsageCollector(plugins.usageCollection);
    return {
      inventoryViews,
      metricsExplorerViews
    };
  }
  start(core) {
    var _this$metricsExplorer2;
    const inventoryViews = this.inventoryViews.start({
      infraSources: this.libs.sources,
      savedObjects: core.savedObjects
    });
    const metricsExplorerViews = (_this$metricsExplorer2 = this.metricsExplorerViews) === null || _this$metricsExplorer2 === void 0 ? void 0 : _this$metricsExplorer2.start({
      infraSources: this.libs.sources,
      savedObjects: core.savedObjects
    });
    (0, _infra_server.registerRoutes)(this.libs);
    return {
      inventoryViews,
      metricsExplorerViews
    };
  }
  stop() {}
}
exports.InfraServerPlugin = InfraServerPlugin;