"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getLogsHasDataFetcher = getLogsHasDataFetcher;
exports.getLogsOverviewDataFetcher = getLogsOverviewDataFetcher;
var _common = require("@kbn/logs-shared-plugin/common");
var _constants = require("../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getLogsHasDataFetcher(getStartServices) {
  return async () => {
    const [coreStart, {
      logsShared
    }] = await getStartServices();
    const resolvedLogView = await logsShared.logViews.client.getResolvedLogView(_common.DEFAULT_LOG_VIEW);
    const logViewStatus = await logsShared.logViews.client.getResolvedLogViewStatus(resolvedLogView, coreStart.uiSettings);
    const hasData = logViewStatus.index === 'available';
    const indices = resolvedLogView.indices;
    return {
      hasData,
      indices
    };
  };
}
function getLogsOverviewDataFetcher(getStartServices) {
  return async params => {
    var _logsLocator$getRedir;
    const [coreStart, {
      data,
      logsShared,
      share
    }] = await getStartServices();
    const resolvedLogView = await logsShared.logViews.client.getResolvedLogView(_common.DEFAULT_LOG_VIEW);
    const {
      stats,
      series
    } = await fetchLogsOverview({
      index: resolvedLogView.indices
    }, params, data, coreStart.uiSettings);
    const {
      logsLocator
    } = (0, _common.getLogsLocatorsFromUrlService)(share.url);
    const timeSpanInMinutes = (params.absoluteTime.end - params.absoluteTime.start) / (1000 * 60);
    const appLink = (_logsLocator$getRedir = logsLocator === null || logsLocator === void 0 ? void 0 : logsLocator.getRedirectUrl({
      timeRange: {
        startTime: params.absoluteTime.start,
        endTime: params.absoluteTime.end
      }
    })) !== null && _logsLocator$getRedir !== void 0 ? _logsLocator$getRedir : '';
    return {
      appLink,
      stats: normalizeStats(stats, timeSpanInMinutes),
      series: normalizeSeries(series)
    };
  };
}
async function fetchLogsOverview(logParams, params, dataPlugin, uiSettings) {
  var _uiSettings$get;
  const excludedDataTiers = (_uiSettings$get = uiSettings === null || uiSettings === void 0 ? void 0 : uiSettings.get('observability:searchExcludedDataTiers')) !== null && _uiSettings$get !== void 0 ? _uiSettings$get : [];
  return new Promise((resolve, reject) => {
    let esResponse;
    dataPlugin.search.search({
      params: {
        index: logParams.index,
        body: {
          size: 0,
          query: buildLogOverviewQuery(params, excludedDataTiers),
          aggs: buildLogOverviewAggregations(logParams, params)
        }
      }
    }).subscribe(response => esResponse = response.rawResponse, error => reject(error), () => {
      var _esResponse;
      if ((_esResponse = esResponse) !== null && _esResponse !== void 0 && _esResponse.aggregations) {
        resolve(processLogsOverviewAggregations(esResponse.aggregations));
      } else {
        resolve({
          stats: {},
          series: {}
        });
      }
    });
  });
}
function buildLogOverviewQuery(params, excludedDataTiers) {
  return {
    bool: {
      must: [{
        range: {
          [_constants.TIMESTAMP_FIELD]: {
            gt: new Date(params.absoluteTime.start).toISOString(),
            lte: new Date(params.absoluteTime.end).toISOString(),
            format: 'strict_date_optional_time'
          }
        }
      }],
      must_not: excludedDataTiers.length ? [{
        terms: {
          _tier: excludedDataTiers
        }
      }] : undefined
    }
  };
}
function buildLogOverviewAggregations(logParams, params) {
  return {
    stats: {
      terms: {
        field: 'event.dataset',
        size: 4,
        missing: 'unknown'
      },
      aggs: {
        series: {
          date_histogram: {
            field: _constants.TIMESTAMP_FIELD,
            fixed_interval: params.intervalString
          }
        }
      }
    }
  };
}
function processLogsOverviewAggregations(aggregations) {
  const processedStats = {};
  const processedSeries = {};
  aggregations.stats.buckets.forEach(stat => {
    const label = stat.key;
    processedStats[stat.key] = {
      type: 'number',
      label,
      value: stat.doc_count
    };
    stat.series.buckets.forEach(series => {
      processedSeries[label] = processedSeries[label] || {
        label,
        coordinates: []
      };
      processedSeries[label].coordinates.push({
        x: series.key,
        y: series.doc_count
      });
    });
  });
  return {
    stats: processedStats,
    series: processedSeries
  };
}
function normalizeStats(stats, timeSpanInMinutes) {
  return Object.keys(stats).reduce((normalized, key) => {
    normalized[key] = {
      ...stats[key],
      value: stats[key].value / timeSpanInMinutes
    };
    return normalized;
  }, {});
}
function normalizeSeries(series) {
  const seriesKeys = Object.keys(series);
  const timestamps = seriesKeys.flatMap(key => series[key].coordinates.map(c => c.x));
  const [first, second] = [...new Set(timestamps)].sort();
  const timeSpanInMinutes = (second - first) / (1000 * 60);
  return seriesKeys.reduce((normalized, key) => {
    normalized[key] = {
      ...series[key],
      coordinates: series[key].coordinates.map(c => {
        if (c.y) {
          return {
            ...c,
            y: c.y / timeSpanInMinutes
          };
        }
        return c;
      })
    };
    return normalized;
  }, {});
}