"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.messagesToInference = void 0;
var _inferenceCommon = require("@kbn/inference-common");
var _messages = require("@langchain/core/messages");
var _langchain = require("../utils/langchain");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// type is not exposed from the lib...

const messagesToInference = messages => {
  return messages.reduce((output, message) => {
    if ((0, _messages.isSystemMessage)(message)) {
      const content = extractMessageTextContent(message);
      output.system = output.system ? `${output.system}\n${content}` : content;
    }
    if ((0, _messages.isHumanMessage)(message)) {
      output.messages.push({
        role: _inferenceCommon.MessageRole.User,
        content: convertMessageContent(message)
      });
    }
    if ((0, _messages.isAIMessage)(message)) {
      var _message$tool_calls, _message$additional_k, _message$additional_k2;
      output.messages.push({
        role: _inferenceCommon.MessageRole.Assistant,
        content: extractMessageTextContent(message),
        toolCalls: (_message$tool_calls = message.tool_calls) !== null && _message$tool_calls !== void 0 && _message$tool_calls.length ? message.tool_calls.map(toolCallToInference) : (_message$additional_k = message.additional_kwargs) !== null && _message$additional_k !== void 0 && (_message$additional_k2 = _message$additional_k.tool_calls) !== null && _message$additional_k2 !== void 0 && _message$additional_k2.length ? message.additional_kwargs.tool_calls.map(legacyToolCallToInference) : undefined
      });
    }
    if ((0, _messages.isToolMessage)(message)) {
      output.messages.push({
        role: _inferenceCommon.MessageRole.Tool,
        // langchain does not have the function name on tool messages
        name: message.tool_call_id,
        toolCallId: message.tool_call_id,
        response: toolResponseContentToInference(message.content)
      });
    }
    if ((0, _messages.isFunctionMessage)(message) && message.additional_kwargs.function_call) {
      output.messages.push({
        role: _inferenceCommon.MessageRole.Tool,
        name: message.additional_kwargs.function_call.name,
        toolCallId: (0, _inferenceCommon.generateFakeToolCallId)(),
        response: toolResponseContentToInference(message.content)
      });
    }
    return output;
  }, {
    messages: [],
    system: undefined
  });
};
exports.messagesToInference = messagesToInference;
const toolResponseContentToInference = toolResponse => {
  const content = typeof toolResponse === 'string' ? toolResponse : toolResponse.filter(_langchain.isMessageContentText).map(part => part.text).join('\n');
  try {
    return JSON.parse(content);
  } catch (e) {
    return {
      response: content
    };
  }
};
const toolCallToInference = toolCall => {
  var _toolCall$id;
  return {
    toolCallId: (_toolCall$id = toolCall.id) !== null && _toolCall$id !== void 0 ? _toolCall$id : (0, _inferenceCommon.generateFakeToolCallId)(),
    function: {
      name: toolCall.name,
      arguments: toolCall.args
    }
  };
};
const legacyToolCallToInference = toolCall => {
  return {
    toolCallId: toolCall.id,
    function: {
      name: toolCall.function.name,
      arguments: {
        args: toolCall.function.arguments
      }
    }
  };
};
const extractMessageTextContent = message => {
  if (typeof message.content === 'string') {
    return message.content;
  }
  return message.content.filter(_langchain.isMessageContentText).map(part => part.text).join('\n');
};
const convertMessageContent = message => {
  if (typeof message.content === 'string') {
    return message.content;
  }
  return message.content.reduce((messages, part) => {
    if ((0, _langchain.isMessageContentText)(part)) {
      messages.push({
        type: 'text',
        text: part.text
      });
    } else if ((0, _langchain.isMessageContentImageUrl)(part)) {
      const imageUrl = typeof part.image_url === 'string' ? part.image_url : part.image_url.url;
      messages.push({
        type: 'image',
        source: {
          data: imageUrl,
          mimeType: ''
        }
      });
    }
    return messages;
  }, []);
};