"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.usePackagePoliciesWithAgentPolicy = void 0;
var _react = require("react");
var _constants = require("../../../../../../../../common/constants");
var _services = require("../../../../../services");
var _hooks = require("../../../../../hooks");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Works similar to `useGetAgentPolicies()`, except that it will add an additional property to
 * each package policy named `agentPolicies` which may hold the Agent Policies associated with the
 * given package policy.
 * @param query
 */
const usePackagePoliciesWithAgentPolicy = query => {
  const {
    data: packagePoliciesData,
    error,
    isLoading: isLoadingPackagePolicies,
    resendRequest
  } = (0, _hooks.useGetPackagePolicies)(query);
  const agentPoliciesIds = (0, _react.useMemo)(() => {
    if (!(packagePoliciesData !== null && packagePoliciesData !== void 0 && packagePoliciesData.items.length)) {
      return [];
    }

    // Build a list of package_policies for which we need Agent Policies for. Since some package
    // policies can exist within the same Agent Policy, we don't need to (in some cases) include
    // the entire list of package_policy ids.
    return Array.from(new Set(packagePoliciesData.items.flatMap(packagePolicy => packagePolicy.policy_ids)).values());
  }, [packagePoliciesData]);
  const {
    data: agentPoliciesData,
    isLoading: isLoadingAgentPolicies
  } = (0, _hooks.useConditionalRequest)({
    path: _services.agentPolicyRouteService.getBulkGetPath(),
    method: 'post',
    body: {
      ids: agentPoliciesIds,
      full: true,
      ignoreMissing: true
    },
    version: _constants.API_VERSIONS.public.v1,
    shouldSendRequest: agentPoliciesIds.length > 0
  });
  const [enrichedData, setEnrichedData] = (0, _react.useState)();
  (0, _react.useEffect)(() => {
    if (isLoadingPackagePolicies || isLoadingAgentPolicies) {
      return;
    }
    if (!(packagePoliciesData !== null && packagePoliciesData !== void 0 && packagePoliciesData.items)) {
      setEnrichedData(undefined);
      return;
    }
    const agentPoliciesById = {};
    if (agentPoliciesData !== null && agentPoliciesData !== void 0 && agentPoliciesData.items) {
      for (const agentPolicy of agentPoliciesData.items) {
        agentPoliciesById[agentPolicy.id] = agentPolicy;
      }
    }
    const updatedPackageData = packagePoliciesData.items.map(packagePolicy => {
      return {
        packagePolicy,
        agentPolicies: packagePolicy.policy_ids.map(policyId => agentPoliciesById[policyId]).filter(policy => !!policy)
      };
    });
    setEnrichedData({
      ...packagePoliciesData,
      items: updatedPackageData
    });
  }, [isLoadingAgentPolicies, isLoadingPackagePolicies, packagePoliciesData, agentPoliciesData]);
  return {
    data: enrichedData,
    error,
    isLoading: isLoadingPackagePolicies || isLoadingAgentPolicies,
    resendRequest
  };
};
exports.usePackagePoliciesWithAgentPolicy = usePackagePoliciesWithAgentPolicy;