"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ignoreErrorsMap = void 0;
exports.validateColumnForCommand = validateColumnForCommand;
exports.validateQuery = validateQuery;
exports.validateSources = validateSources;
var _esqlAst = require("@kbn/esql-ast");
var _helpers = require("../shared/helpers");
var _user_defined_columns = require("../shared/user_defined_columns");
var _errors = require("./errors");
var _function_validation = require("./function_validation");
var _resources = require("./resources");
var _join = require("./commands/join");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * ES|QL validation public API
 * It takes a query string and returns a list of messages (errors and warnings) after validate
 * The astProvider is optional, but if not provided the default one from '@kbn/esql-validation-autocomplete' will be used.
 * This is useful for async loading the ES|QL parser and reduce the bundle size, or to swap grammar version.
 * As for the callbacks, while optional, the validation function will selectively ignore some errors types based on each callback missing.
 */
async function validateQuery(queryString, options = {}, callbacks) {
  const result = await validateAst(queryString, callbacks);
  // early return if we do not want to ignore errors
  if (!options.ignoreOnMissingCallbacks) {
    return result;
  }
  const finalCallbacks = callbacks || {};
  const errorTypoesToIgnore = Object.entries(ignoreErrorsMap).reduce((acc, [key, errorCodes]) => {
    if (!(key in finalCallbacks) || key in finalCallbacks && finalCallbacks[key] == null) {
      for (const e of errorCodes) {
        acc[e] = true;
      }
    }
    return acc;
  }, {});
  const filteredErrors = result.errors.filter(error => {
    if ('severity' in error) {
      return true;
    }
    return !errorTypoesToIgnore[error.code];
  }).map(error => 'severity' in error ? {
    text: error.message,
    code: error.code,
    type: 'error',
    location: {
      min: error.startColumn,
      max: error.endColumn
    }
  } : error);
  return {
    errors: filteredErrors,
    warnings: result.warnings
  };
}

/**
 * @internal
 */
const ignoreErrorsMap = exports.ignoreErrorsMap = {
  getColumnsFor: ['unknownColumn', 'wrongArgumentType', 'unsupportedFieldType'],
  getSources: ['unknownIndex'],
  getPolicies: ['unknownPolicy'],
  getPreferences: [],
  getFieldsMetadata: [],
  getVariables: [],
  canSuggestVariables: [],
  getJoinIndices: [],
  getTimeseriesIndices: [],
  getEditorExtensions: [],
  getInferenceEndpoints: []
};

/**
 * This function will perform an high level validation of the
 * query AST. An initial syntax validation is already performed by the parser
 * while here it can detect things like function names, types correctness and potential warnings
 * @param ast A valid AST data structure
 */
async function validateAst(queryString, callbacks) {
  var _callbacks$getJoinInd;
  const messages = [];
  const parsingResult = (0, _esqlAst.parse)(queryString);
  const {
    ast
  } = parsingResult;
  const [sources, availableFields, availablePolicies, joinIndices] = await Promise.all([
  // retrieve the list of available sources
  (0, _resources.retrieveSources)(ast, callbacks),
  // retrieve available fields (if a source command has been defined)
  (0, _resources.retrieveFields)(queryString, ast, callbacks),
  // retrieve available policies (if an enrich command has been defined)
  (0, _resources.retrievePolicies)(ast, callbacks), // retrieve indices for join command
  callbacks === null || callbacks === void 0 ? void 0 : (_callbacks$getJoinInd = callbacks.getJoinIndices) === null || _callbacks$getJoinInd === void 0 ? void 0 : _callbacks$getJoinInd.call(callbacks)]);
  if (availablePolicies.size) {
    const fieldsFromPoliciesMap = await (0, _resources.retrievePoliciesFields)(ast, availablePolicies, callbacks);
    fieldsFromPoliciesMap.forEach((value, key) => availableFields.set(key, value));
  }
  if (ast.some(({
    name
  }) => ['grok', 'dissect'].includes(name))) {
    const fieldsFromGrokOrDissect = await (0, _resources.retrieveFieldsFromStringSources)(queryString, ast, callbacks);
    fieldsFromGrokOrDissect.forEach((value, key) => {
      // if the field is already present, do not overwrite it
      // Note: this can also overlap with some userDefinedColumns
      if (!availableFields.has(key)) {
        availableFields.set(key, value);
      }
    });
  }
  const userDefinedColumns = (0, _user_defined_columns.collectUserDefinedColumns)(ast, availableFields, queryString);
  // notify if the user is rewriting a column as userDefinedColumn with another type
  messages.push(...validateFieldsShadowing(availableFields, userDefinedColumns));
  messages.push(...validateUnsupportedTypeFields(availableFields, ast));
  const references = {
    sources,
    fields: availableFields,
    policies: availablePolicies,
    userDefinedColumns,
    query: queryString,
    joinIndices: (joinIndices === null || joinIndices === void 0 ? void 0 : joinIndices.indices) || []
  };
  let seenFork = false;
  for (const [index, command] of ast.entries()) {
    if (command.name === 'fork') {
      if (seenFork) {
        messages.push(_errors.errors.tooManyForks(command));
      } else {
        seenFork = true;
      }
    }
    const commandMessages = validateCommand(command, references, ast, index);
    messages.push(...commandMessages);
  }
  const parserErrors = parsingResult.errors;

  /**
   * Some changes to the grammar deleted the literal names for some tokens.
   * This is a workaround to restore the literals that were lost.
   *
   * See https://github.com/elastic/elasticsearch/pull/124177 for context.
   */
  for (const error of parserErrors) {
    error.message = error.message.replace(/\bLP\b/, "'('");
    error.message = error.message.replace(/\bOPENING_BRACKET\b/, "'['");
  }
  return {
    errors: [...parserErrors, ...messages.filter(({
      type
    }) => type === 'error')],
    warnings: messages.filter(({
      type
    }) => type === 'warning')
  };
}
function validateCommand(command, references, ast, currentCommandIndex) {
  const messages = [];
  if (command.incomplete) {
    return messages;
  }
  // do not check the command exists, the grammar is already picking that up
  const commandDef = (0, _helpers.getCommandDefinition)(command.name);
  if (!commandDef) {
    return messages;
  }
  if (commandDef.validate) {
    messages.push(...commandDef.validate(command, references, ast));
  }
  switch (commandDef.name) {
    case 'join':
      {
        const join = command;
        const joinCommandErrors = (0, _join.validate)(join, references);
        messages.push(...joinCommandErrors);
        break;
      }
    case 'fork':
      {
        references.fields.set('_fork', {
          name: '_fork',
          type: 'keyword'
        });
        for (const arg of command.args.flat()) {
          if ((0, _helpers.isSingleItem)(arg) && arg.type === 'query') {
            // all the args should be commands
            arg.commands.forEach(subCommand => {
              messages.push(...validateCommand(subCommand, references, ast, currentCommandIndex));
            });
          }
        }
      }
    default:
      {
        // Now validate arguments
        for (const arg of command.args) {
          if (!Array.isArray(arg)) {
            if ((0, _helpers.isFunctionItem)(arg)) {
              messages.push(...(0, _function_validation.validateFunction)({
                fn: arg,
                parentCommand: command.name,
                parentOption: undefined,
                references,
                parentAst: ast,
                currentCommandIndex
              }));
            } else if ((0, _helpers.isOptionItem)(arg)) {
              messages.push(...validateOption(arg, command, references));
            } else if ((0, _helpers.isColumnItem)(arg) || (0, _esqlAst.isIdentifier)(arg)) {
              if (command.name === 'stats' || command.name === 'inlinestats') {
                messages.push(_errors.errors.unknownAggFunction(arg));
              } else {
                messages.push(...validateColumnForCommand(arg, command.name, references));
              }
            } else if ((0, _helpers.isTimeIntervalItem)(arg)) {
              messages.push((0, _errors.getMessageFromId)({
                messageId: 'unsupportedTypeForCommand',
                values: {
                  command: command.name.toUpperCase(),
                  type: 'date_period',
                  value: arg.name
                },
                locations: arg.location
              }));
            }
          }
        }
        const sources = command.args.filter(arg => (0, _helpers.isSourceItem)(arg));
        messages.push(...validateSources(sources, references));
      }
  }

  // no need to check for mandatory options passed
  // as they are already validated at syntax level
  return messages;
}
function validateOption(option, command, referenceMaps) {
  // check if the arguments of the option are of the correct type
  const messages = [];
  if (option.incomplete || command.incomplete || option.name === 'metadata') {
    return messages;
  }
  if (option.name === 'metadata') {
    // Validation for the metadata statement is handled in the FROM command's validate method
    return messages;
  }
  for (const arg of option.args) {
    if (Array.isArray(arg)) {
      continue;
    }
    if ((0, _helpers.isColumnItem)(arg)) {
      messages.push(...validateColumnForCommand(arg, command.name, referenceMaps));
    } else if ((0, _helpers.isFunctionItem)(arg)) {
      messages.push(...(0, _function_validation.validateFunction)({
        fn: arg,
        parentCommand: command.name,
        parentOption: option.name,
        references: referenceMaps
      }));
    }
  }
  return messages;
}
function validateFieldsShadowing(fields, userDefinedColumns) {
  const messages = [];
  for (const userDefinedColumn of userDefinedColumns.keys()) {
    if (fields.has(userDefinedColumn)) {
      var _fields$get;
      const userDefinedColumnHits = userDefinedColumns.get(userDefinedColumn);
      if (!(0, _helpers.areFieldAndUserDefinedColumnTypesCompatible)((_fields$get = fields.get(userDefinedColumn)) === null || _fields$get === void 0 ? void 0 : _fields$get.type, userDefinedColumnHits[0].type)) {
        const fieldType = fields.get(userDefinedColumn).type;
        const userDefinedColumnType = userDefinedColumnHits[0].type;
        const flatFieldType = fieldType;
        const flatUserDefinedColumnType = userDefinedColumnType;
        messages.push((0, _errors.getMessageFromId)({
          messageId: 'shadowFieldType',
          values: {
            field: userDefinedColumn,
            fieldType: flatFieldType,
            newType: flatUserDefinedColumnType
          },
          locations: userDefinedColumnHits[0].location
        }));
      }
    }
  }
  return messages;
}
function validateUnsupportedTypeFields(fields, ast) {
  const usedColumnsInQuery = [];
  (0, _esqlAst.walk)(ast, {
    visitColumn: node => usedColumnsInQuery.push(node.name)
  });
  const messages = [];
  for (const column of usedColumnsInQuery) {
    if (fields.has(column) && fields.get(column).type === 'unsupported') {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'unsupportedFieldType',
        values: {
          field: column
        },
        locations: {
          min: 1,
          max: 1
        }
      }));
    }
  }
  return messages;
}
function validateSources(sources, {
  sources: availableSources
}) {
  const messages = [];
  const knownIndexNames = [];
  const knownIndexPatterns = [];
  const unknownIndexNames = [];
  const unknownIndexPatterns = [];
  for (const source of sources) {
    if (source.incomplete) {
      return messages;
    }
    if (source.sourceType === 'index') {
      const index = source.index;
      const sourceName = source.prefix ? source.name : index === null || index === void 0 ? void 0 : index.valueUnquoted;
      if (!sourceName) continue;
      if ((0, _helpers.sourceExists)(sourceName, availableSources) && !(0, _helpers.hasWildcard)(sourceName)) {
        knownIndexNames.push(source);
      }
      if ((0, _helpers.sourceExists)(sourceName, availableSources) && (0, _helpers.hasWildcard)(sourceName)) {
        knownIndexPatterns.push(source);
      }
      if (!(0, _helpers.sourceExists)(sourceName, availableSources) && !(0, _helpers.hasWildcard)(sourceName)) {
        unknownIndexNames.push(source);
      }
      if (!(0, _helpers.sourceExists)(sourceName, availableSources) && (0, _helpers.hasWildcard)(sourceName)) {
        unknownIndexPatterns.push(source);
      }
    }
  }
  unknownIndexNames.forEach(source => {
    messages.push((0, _errors.getMessageFromId)({
      messageId: 'unknownIndex',
      values: {
        name: source.name
      },
      locations: source.location
    }));
  });
  if (knownIndexNames.length + unknownIndexNames.length + knownIndexPatterns.length === 0) {
    // only if there are no known index names, no known index patterns, and no unknown
    // index names do we worry about creating errors for unknown index patterns
    unknownIndexPatterns.forEach(source => {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'unknownIndex',
        values: {
          name: source.name
        },
        locations: source.location
      }));
    });
  }
  return messages;
}
function validateColumnForCommand(column, commandName, references) {
  const messages = [];
  if (commandName === 'row') {
    if (!references.userDefinedColumns.has(column.name) && !(0, _helpers.isParametrized)(column)) {
      messages.push(_errors.errors.unknownColumn(column));
    }
  } else if (!(0, _helpers.getColumnExists)(column, references) && !(0, _helpers.isParametrized)(column)) {
    messages.push(_errors.errors.unknownColumn(column));
  }
  return messages;
}