"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.suggest = void 0;
var _i18n = require("@kbn/i18n");
var _esqlAst = require("@kbn/esql-ast");
var _helpers = require("../../../shared/helpers");
var _helper = require("../../helper");
var _util = require("./util");
var _factories = require("../../factories");
var _complete_items = require("../../complete_items");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const getFullCommandMnemonics = definition => {
  var _definition$types;
  const types = (_definition$types = definition.types) !== null && _definition$types !== void 0 ? _definition$types : [];
  if (!types.length) {
    return [[definition.name, definition.description]];
  }
  return types.map(type => {
    var _type$description;
    return [`${type.name.toUpperCase()} ${definition.name.toUpperCase()}`, (_type$description = type.description) !== null && _type$description !== void 0 ? _type$description : definition.description];
  });
};

// facilitates fast checks for the existence of fields in the lookup index
// by caching the fields of the last lookup index pattern
const lookupIndexFieldSet = {
  set: new Set(),
  key: ''
};
const getLookupFields = async (command, callbacks) => {
  if (!callbacks.getColumnsFor) {
    return [];
  }
  const summary = _esqlAst.mutate.commands.join.summarizeCommand(command);
  const joinIndexPattern = _esqlAst.LeafPrinter.print(summary.target.index);
  const columns = await callbacks.getColumnsFor({
    query: `FROM ${joinIndexPattern}`
  });
  if (lookupIndexFieldSet.key !== joinIndexPattern) {
    lookupIndexFieldSet.set = new Set(columns.map(c => c.name));
    lookupIndexFieldSet.key = joinIndexPattern;
  }
  return columns;
};
const getFieldSuggestions = async (command, getColumnsByType, callbacks) => {
  var _callbacks$canSuggest, _callbacks$canSuggest2;
  if (!callbacks) {
    return {
      suggestions: [],
      lookupIndexFieldExists: () => false
    };
  }
  const onOption = command.args.find(arg => (0, _helpers.isSingleItem)(arg) && arg.name === 'on');
  const ignoredFields = onOption.args.map(arg => (0, _helpers.isColumnItem)(arg) ? arg.parts.join('.') : '');
  const [lookupIndexFields, sourceFields] = await Promise.all([getLookupFields(command, callbacks), getColumnsByType(['any'], ignoredFields, {
    advanceCursor: false,
    openSuggestions: true
  })]);
  const supportsControls = (_callbacks$canSuggest = callbacks === null || callbacks === void 0 ? void 0 : (_callbacks$canSuggest2 = callbacks.canSuggestVariables) === null || _callbacks$canSuggest2 === void 0 ? void 0 : _callbacks$canSuggest2.call(callbacks)) !== null && _callbacks$canSuggest !== void 0 ? _callbacks$canSuggest : false;
  const getVariables = callbacks === null || callbacks === void 0 ? void 0 : callbacks.getVariables;
  const joinFields = (0, _factories.buildFieldsDefinitionsWithMetadata)(lookupIndexFields.filter(f => !ignoredFields.includes(f.name)), [], {
    supportsControls
  }, getVariables);
  const intersection = (0, _util.suggestionIntersection)(joinFields, sourceFields);
  const union = (0, _util.suggestionUnion)(sourceFields, joinFields);
  for (const commonField of intersection) {
    commonField.sortText = '1';
    commonField.documentation = {
      value: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.join.sharedField', {
        defaultMessage: 'Field shared between the source and the lookup index'
      })
    };
    let detail = commonField.detail || '';
    if (detail) {
      detail += ' ';
    }
    detail += _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.join.commonFieldNote', {
      defaultMessage: '(common field)'
    });
    commonField.detail = detail;
  }
  return {
    suggestions: [...intersection, ...union],
    lookupIndexFieldExists: field => lookupIndexFieldSet.set.has((0, _helpers.unescapeColumnName)(field))
  };
};
const suggestFields = async (innerText, command, getColumnsByType, callbacks, columnExists) => {
  if (!callbacks) {
    return [];
  }
  const {
    suggestions: fieldSuggestions,
    lookupIndexFieldExists
  } = await getFieldSuggestions(command, getColumnsByType, callbacks);
  return (0, _helper.handleFragment)(innerText, fragment => columnExists(fragment) || lookupIndexFieldExists(fragment), (_fragment, rangeToReplace) => {
    // fie<suggest>
    return fieldSuggestions.map(suggestion => {
      return {
        ...suggestion,
        text: suggestion.text,
        command: _factories.TRIGGER_SUGGESTION_COMMAND,
        rangeToReplace
      };
    });
  }, (fragment, rangeToReplace) => {
    // field<suggest>
    const finalSuggestions = [{
      ..._complete_items.pipeCompleteItem,
      text: ' | '
    }];
    // when we fix the editor marker, this should probably be checked against 0 instead of 1
    // this is because the last field in the AST is currently getting removed (because it contains
    // the editor marker) so it is not included in the ignored list which is used to filter out
    // existing fields above.
    if (fieldSuggestions.length > 1) finalSuggestions.push({
      ..._complete_items.commaCompleteItem,
      text: ', '
    });
    return finalSuggestions.map(s => ({
      ...s,
      filterText: fragment,
      text: fragment + s.text,
      command: _factories.TRIGGER_SUGGESTION_COMMAND,
      rangeToReplace
    }));
  });
};
const suggest = async ({
  innerText,
  command,
  getColumnsByType,
  columnExists,
  definition,
  callbacks
}) => {
  let commandText = innerText;
  if (command.location) {
    commandText = innerText.slice(command.location.min);
  }
  const position = (0, _util.getPosition)(commandText, command);
  switch (position.pos) {
    case 'type':
    case 'after_type':
    case 'mnemonic':
      {
        const allMnemonics = getFullCommandMnemonics(definition);
        const filteredMnemonics = allMnemonics.filter(([mnemonic]) => mnemonic.startsWith(commandText.toUpperCase()));
        if (!filteredMnemonics.length) {
          return [];
        }
        return filteredMnemonics.map(([mnemonic, description], i) => ({
          label: mnemonic,
          text: mnemonic + ' $0',
          detail: description,
          kind: 'Keyword',
          sortText: `${i}-MNEMONIC`,
          command: _factories.TRIGGER_SUGGESTION_COMMAND
        }));
      }
    case 'after_mnemonic':
    case 'index':
      {
        var _callbacks$getJoinInd;
        const joinIndices = await (callbacks === null || callbacks === void 0 ? void 0 : (_callbacks$getJoinInd = callbacks.getJoinIndices) === null || _callbacks$getJoinInd === void 0 ? void 0 : _callbacks$getJoinInd.call(callbacks));
        if (!joinIndices) {
          return [];
        }
        return (0, _helper.specialIndicesToSuggestions)(joinIndices.indices);
      }
    case 'after_index':
      {
        const suggestion = {
          label: 'ON',
          text: 'ON ',
          detail: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.join.onKeyword', {
            defaultMessage: 'Specify JOIN field conditions'
          }),
          kind: 'Keyword',
          sortText: '0-ON',
          command: _factories.TRIGGER_SUGGESTION_COMMAND
        };
        return [suggestion];
      }
    case 'after_on':
      {
        return suggestFields(innerText, command, getColumnsByType, callbacks, columnExists);
      }
    case 'condition':
      {
        if (/(?<!\,)\s+$/.test(innerText)) {
          // this trailing whitespace was not proceeded by a comma
          return [_complete_items.commaCompleteItem, _complete_items.pipeCompleteItem];
        }
        const fields = await suggestFields(innerText, command, getColumnsByType, callbacks, columnExists);
        return fields;
      }
  }
  const suggestions = [];
  return suggestions;
};
exports.suggest = suggest;