"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isValidExternalUrl = exports.ExternalConnectorLogic = void 0;
var _kea = require("kea");
var _i18n = require("@kbn/i18n");
var _flash_messages = require("../../../../../../shared/flash_messages");
var _http = require("../../../../../../shared/http");
var _kibana = require("../../../../../../shared/kibana");
var _app_logic = require("../../../../../app_logic");
var _routes = require("../../../../../routes");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ExternalConnectorLogic = exports.ExternalConnectorLogic = (0, _kea.kea)({
  path: ['enterprise_search', 'workplace_search', 'external_connector_logic'],
  actions: {
    fetchExternalSource: true,
    fetchExternalSourceSuccess: sourceConfigData => sourceConfigData,
    saveExternalConnectorConfigError: () => true,
    saveExternalConnectorConfigSuccess: externalConnectorId => externalConnectorId,
    saveExternalConnectorConfig: () => true,
    setExternalConnectorApiKey: externalConnectorApiKey => externalConnectorApiKey,
    setExternalConnectorUrl: externalConnectorUrl => externalConnectorUrl,
    setUrlValidation: valid => valid,
    setShowInsecureUrlCallout: showCallout => showCallout,
    validateUrl: true
  },
  reducers: {
    dataLoading: [true, {
      fetchExternalSourceSuccess: () => false
    }],
    buttonLoading: [false, {
      saveExternalConnectorConfigSuccess: () => false,
      saveExternalConnectorConfigError: () => false,
      saveExternalConnectorConfig: () => true
    }],
    externalConnectorUrl: ['', {
      fetchExternalSourceSuccess: (_, {
        configuredFields: {
          external_connector_url: externalConnectorUrl
        }
      }) => externalConnectorUrl || '',
      setExternalConnectorUrl: (_, url) => url
    }],
    externalConnectorApiKey: ['', {
      fetchExternalSourceSuccess: (_, {
        configuredFields: {
          external_connector_api_key: externalConnectorApiKey
        }
      }) => externalConnectorApiKey || '',
      setExternalConnectorApiKey: (_, apiKey) => apiKey
    }],
    showInsecureUrlCallout: [false, {
      fetchExternalSource: () => false,
      setShowInsecureUrlCallout: (_, showCallout) => showCallout
    }],
    urlValid: [true, {
      setUrlValidation: (_, valid) => valid
    }]
  },
  listeners: ({
    actions,
    values
  }) => ({
    fetchExternalSource: async () => {
      const route = '/internal/workplace_search/org/settings/connectors/external';
      try {
        const response = await _http.HttpLogic.values.http.get(route);
        actions.fetchExternalSourceSuccess(response);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    fetchExternalSourceSuccess: ({
      configuredFields: {
        externalConnectorUrl
      }
    }) => {
      if (externalConnectorUrl && !externalConnectorUrl.startsWith('https://')) {
        actions.setShowInsecureUrlCallout(true);
      } else {
        actions.setShowInsecureUrlCallout(false);
      }
    },
    saveExternalConnectorConfig: async () => {
      if (!isValidExternalUrl(values.externalConnectorUrl)) {
        actions.setUrlValidation(false);
      } else {
        (0, _flash_messages.clearFlashMessages)();
        try {
          await _http.HttpLogic.values.http.post('/internal/workplace_search/org/settings/connectors', {
            body: JSON.stringify({
              external_connector_url: values.externalConnectorUrl,
              external_connector_api_key: values.externalConnectorApiKey,
              service_type: 'external'
            })
          });
          (0, _flash_messages.flashSuccessToast)(_i18n.i18n.translate('xpack.enterpriseSearch.workplaceSearch.sources.flashMessages.externalConnectorCreated', {
            defaultMessage: 'Successfully registered connector package deployment.'
          }));
          // TODO: Once we have multiple external connector types, use response data instead
          actions.saveExternalConnectorConfigSuccess('external');
          _kibana.KibanaLogic.values.navigateToUrl((0, _routes.getSourcesPath)(`${(0, _routes.getAddPath)('external')}`, _app_logic.AppLogic.values.isOrganization));
        } catch (e) {
          actions.saveExternalConnectorConfigError();
          (0, _flash_messages.flashAPIErrors)(e);
        }
      }
    },
    validateUrl: () => {
      const url = values.externalConnectorUrl;
      actions.setUrlValidation(isValidExternalUrl(url));
      actions.setShowInsecureUrlCallout(!url.startsWith('https://'));
    }
  }),
  selectors: ({
    selectors
  }) => ({
    formDisabled: [() => [selectors.buttonLoading, selectors.dataLoading], (buttonLoading, dataLoading) => buttonLoading || dataLoading],
    insecureUrl: [() => [selectors.externalConnectorUrl], url => !url.startsWith('https://')]
  })
});
const isValidExternalUrl = url => url.startsWith('https://') || url.startsWith('http://');
exports.isValidExternalUrl = isValidExternalUrl;