"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.crawlConfigServerToClient = crawlConfigServerToClient;
exports.crawlDomainValidationToResult = crawlDomainValidationToResult;
exports.crawlEventServerToClient = crawlEventServerToClient;
exports.crawlRequestServerToClient = crawlRequestServerToClient;
exports.crawlRequestStatsServerToClient = crawlRequestStatsServerToClient;
exports.crawlRequestWithDetailsServerToClient = crawlRequestWithDetailsServerToClient;
exports.crawlerCustomSchedulingServerToClient = exports.crawlerCustomSchedulingClientToServer = exports.crawlScheduleServerToClient = void 0;
exports.crawlerDataServerToClient = crawlerDataServerToClient;
exports.crawlerDomainServerToClient = crawlerDomainServerToClient;
exports.domainConfigServerToClient = exports.crawlerDomainsWithMetaServerToClient = void 0;
exports.isBasicCrawlerAuth = isBasicCrawlerAuth;
exports.isRawCrawlerAuth = isRawCrawlerAuth;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function crawlerDomainServerToClient(payload) {
  const {
    auth,
    available_deduplication_fields: availableDeduplicationFields,
    crawl_rules: crawlRules,
    created_on: createdOn,
    deduplication_enabled: deduplicationEnabled,
    deduplication_fields: deduplicationFields,
    default_crawl_rule: defaultCrawlRule,
    document_count: documentCount,
    entry_points: entryPoints,
    extraction_rules: extractionRules,
    id,
    last_visited_at: lastCrawl,
    name,
    sitemaps
  } = payload;
  const clientPayload = {
    auth,
    availableDeduplicationFields,
    crawlRules,
    createdOn,
    deduplicationEnabled,
    deduplicationFields,
    documentCount,
    entryPoints,
    extractionRules,
    id,
    sitemaps,
    url: name
  };
  if (lastCrawl) {
    clientPayload.lastCrawl = lastCrawl;
  }
  if (defaultCrawlRule) {
    clientPayload.defaultCrawlRule = defaultCrawlRule;
  }
  return clientPayload;
}
function crawlRequestStatsServerToClient(crawlStats) {
  const {
    status: {
      avg_response_time_msec: avgResponseTimeMSec,
      crawl_duration_msec: crawlDurationMSec,
      pages_visited: pagesVisited,
      urls_allowed: urlsAllowed,
      status_codes: statusCodes
    }
  } = crawlStats;
  return {
    status: {
      avgResponseTimeMSec,
      crawlDurationMSec,
      pagesVisited,
      statusCodes,
      urlsAllowed
    }
  };
}
function crawlRequestServerToClient(crawlRequest) {
  const {
    id,
    status,
    created_at: createdAt,
    began_at: beganAt,
    completed_at: completedAt
  } = crawlRequest;
  return {
    beganAt,
    completedAt,
    createdAt,
    id,
    status
  };
}
function crawlConfigServerToClient(crawlConfig) {
  const {
    domain_allowlist: domainAllowlist,
    seed_urls: seedUrls,
    sitemap_urls: sitemapUrls,
    max_crawl_depth: maxCrawlDepth
  } = crawlConfig;
  return {
    domainAllowlist,
    maxCrawlDepth,
    seedUrls,
    sitemapUrls
  };
}
function crawlEventServerToClient(event) {
  const {
    id,
    stage,
    status,
    created_at: createdAt,
    began_at: beganAt,
    completed_at: completedAt,
    type,
    crawl_config: crawlConfig
  } = event;
  return {
    beganAt,
    completedAt,
    crawlConfig: crawlConfigServerToClient(crawlConfig),
    createdAt,
    id,
    stage,
    status,
    type
  };
}
function crawlRequestWithDetailsServerToClient(event) {
  const {
    began_at: beganAt,
    completed_at: completedAt,
    crawl_config: crawlConfig,
    created_at: createdAt,
    id,
    stats: crawlStats,
    status,
    type
  } = event;
  return {
    beganAt,
    completedAt,
    crawlConfig: crawlConfigServerToClient(crawlConfig),
    createdAt,
    id,
    stats: crawlStats && crawlRequestStatsServerToClient(crawlStats),
    status,
    type
  };
}
function crawlerDataServerToClient(payload) {
  const {
    domains,
    events,
    most_recent_crawl_request: mostRecentCrawlRequest,
    user_agent: userAgent
  } = payload;
  return {
    domains: domains.map(domain => crawlerDomainServerToClient(domain)),
    events: events.map(event => crawlEventServerToClient(event)),
    mostRecentCrawlRequest: mostRecentCrawlRequest && crawlRequestServerToClient(mostRecentCrawlRequest),
    userAgent
  };
}
function crawlDomainValidationToResult(data) {
  if (!data.valid) {
    var _data$results$find;
    return {
      blockingFailure: true,
      message: (_data$results$find = data.results.find(result => result.result === 'failure')) === null || _data$results$find === void 0 ? void 0 : _data$results$find.comment,
      state: 'invalid'
    };
  }
  const warningResult = data.results.find(result => result.result === 'warning');
  if (warningResult) {
    return {
      blockingFailure: !data.valid,
      message: warningResult.comment,
      state: 'warning'
    };
  }
  return {
    state: 'valid'
  };
}
const domainConfigServerToClient = domainConfigFromServer => ({
  id: domainConfigFromServer.id,
  name: domainConfigFromServer.name,
  seedUrls: domainConfigFromServer.seed_urls,
  sitemapUrls: domainConfigFromServer.sitemap_urls
});
exports.domainConfigServerToClient = domainConfigServerToClient;
const crawlerCustomSchedulingServerToClient = customSchedulingFromServer => Object.entries(customSchedulingFromServer.custom_scheduling).map(([scheduleKey, scheduleMapping]) => {
  const {
    name,
    interval,
    configuration_overrides: configurationOverrides,
    enabled
  } = scheduleMapping;
  const {
    max_crawl_depth: maxCrawlDepth = 2,
    sitemap_discovery_disabled: notIncludeSitemapsInRobotsTxt = false,
    domain_allowlist: selectedDomainUrls = [],
    sitemap_urls: customSitemapUrls = [],
    seed_urls: customEntryPointUrls = []
  } = configurationOverrides;
  return {
    scheduleKey,
    name,
    interval,
    enabled,
    maxCrawlDepth,
    includeSitemapsInRobotsTxt: !notIncludeSitemapsInRobotsTxt,
    selectedDomainUrls,
    selectedEntryPointUrls: [],
    selectedSitemapUrls: [],
    customEntryPointUrls,
    customSitemapUrls,
    entryPointUrls: [],
    sitemapUrls: []
  };
});
exports.crawlerCustomSchedulingServerToClient = crawlerCustomSchedulingServerToClient;
const crawlerCustomSchedulingClientToServer = crawlerCustomSchedules => {
  const mapToServerFormat = crawlerSchedule => {
    const configurationOverrides = {
      maxCrawlDepth: crawlerSchedule.maxCrawlDepth,
      sitemapDiscoveryDisabled: !crawlerSchedule.includeSitemapsInRobotsTxt,
      domainAllowlist: crawlerSchedule.selectedDomainUrls,
      sitemapUrls: [...crawlerSchedule.selectedSitemapUrls, ...crawlerSchedule.customSitemapUrls],
      seedUrls: [...crawlerSchedule.selectedEntryPointUrls, ...crawlerSchedule.customEntryPointUrls]
    };
    return {
      name: crawlerSchedule.name,
      interval: crawlerSchedule.interval,
      configurationOverrides,
      enabled: crawlerSchedule.enabled
    };
  };
  const customSchedules = crawlerCustomSchedules.reduce((map, schedule) => {
    map.set(schedule.scheduleKey, mapToServerFormat(schedule));
    return map;
  }, new Map());
  return customSchedules;
};
exports.crawlerCustomSchedulingClientToServer = crawlerCustomSchedulingClientToServer;
const crawlerDomainsWithMetaServerToClient = ({
  results,
  meta
}) => ({
  domains: results.map(crawlerDomainServerToClient),
  meta
});
exports.crawlerDomainsWithMetaServerToClient = crawlerDomainsWithMetaServerToClient;
const crawlScheduleServerToClient = ({
  frequency,
  unit,
  use_connector_schedule: useConnectorSchedule
}) => ({
  frequency,
  unit,
  useConnectorSchedule
});
exports.crawlScheduleServerToClient = crawlScheduleServerToClient;
function isBasicCrawlerAuth(auth) {
  return auth !== null && auth.type === 'basic';
}
function isRawCrawlerAuth(auth) {
  return auth !== null && auth.type === 'raw';
}