"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DocumentDetailLogic = void 0;
var _kea = require("kea");
var _i18n = require("@kbn/i18n");
var _flash_messages = require("../../../shared/flash_messages");
var _http = require("../../../shared/http");
var _kibana = require("../../../shared/kibana");
var _routes = require("../../routes");
var _engine = require("../engine");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DocumentDetailLogic = exports.DocumentDetailLogic = (0, _kea.kea)({
  path: ['enterprise_search', 'app_search', 'document_detail_logic'],
  actions: () => ({
    setFields: fields => ({
      fields
    }),
    getDocumentDetails: documentId => ({
      documentId
    }),
    deleteDocument: documentId => ({
      documentId
    })
  }),
  reducers: () => ({
    dataLoading: [true, {
      setFields: () => false
    }],
    fields: [[], {
      setFields: (_, {
        fields
      }) => fields
    }]
  }),
  listeners: ({
    actions
  }) => ({
    getDocumentDetails: async ({
      documentId
    }) => {
      const {
        engineName
      } = _engine.EngineLogic.values;
      const {
        navigateToUrl
      } = _kibana.KibanaLogic.values;
      try {
        const {
          http
        } = _http.HttpLogic.values;
        const response = await http.get(`/internal/app_search/engines/${engineName}/documents/${documentId}`);
        actions.setFields(response.fields);
      } catch (e) {
        // If an error occurs trying to load this document, it will typically be a 404, or some other
        // error that will prevent the page from loading, so redirect to the documents page and
        // show the error
        (0, _flash_messages.flashAPIErrors)(e, {
          isQueued: true
        });
        navigateToUrl((0, _engine.generateEnginePath)(_routes.ENGINE_DOCUMENTS_PATH));
      }
    },
    deleteDocument: async ({
      documentId
    }) => {
      const {
        engineName
      } = _engine.EngineLogic.values;
      const {
        navigateToUrl
      } = _kibana.KibanaLogic.values;
      const CONFIRM_DELETE = _i18n.i18n.translate('xpack.enterpriseSearch.appSearch.documentDetail.confirmDelete', {
        defaultMessage: 'Are you sure you want to delete this document?'
      });
      const DELETE_SUCCESS = _i18n.i18n.translate('xpack.enterpriseSearch.appSearch.documentDetail.deleteSuccess', {
        defaultMessage: 'Your document was deleted'
      });
      if (window.confirm(CONFIRM_DELETE)) {
        try {
          const {
            http
          } = _http.HttpLogic.values;
          await http.delete(`/internal/app_search/engines/${engineName}/documents/${documentId}`);
          (0, _flash_messages.flashSuccessToast)(DELETE_SUCCESS);
          navigateToUrl((0, _engine.generateEnginePath)(_routes.ENGINE_DOCUMENTS_PATH));
        } catch (e) {
          (0, _flash_messages.flashAPIErrors)(e);
        }
      }
    }
  })
});