"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAttackDiscoveryGenerations = void 0;
var _fp = require("lodash/fp");
var _combine_generations_with_success_metadata = require("../combine_generations_with_success_metadata");
var _get_successfull_generations_query = require("../get_successfull_generations_query");
var _transform_get_attack_discovery_generations_search_result = require("../transforms/transform_get_attack_discovery_generations_search_result");
var _transform_successful_generations_search_result = require("../transforms/transform_successful_generations_search_result");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getAttackDiscoveryGenerations = async ({
  authenticatedUser,
  esClient,
  eventLogIndex,
  generationsQuery,
  getAttackDiscoveryGenerationsParams,
  logger,
  spaceId
}) => {
  const {
    size,
    start,
    end
  } = getAttackDiscoveryGenerationsParams;
  const successfulGenerationsQuery = (0, _get_successfull_generations_query.getSuccessfulGenerationsQuery)({
    authenticatedUser,
    end,
    eventLogIndex,
    size,
    spaceId,
    start
  });

  // Build an msearch query to get both generations and successful generations
  // in a single request, where:
  // 1. The first search is for the generations
  // 2. The second search is for the successful generations (metadata)
  const mSearchQueryBody = {
    searches: [{
      allow_no_indices: generationsQuery.allow_no_indices,
      index: eventLogIndex,
      ignore_unavailable: generationsQuery.ignore_unavailable
    }, {
      aggs: generationsQuery.aggs,
      query: generationsQuery.query,
      size: generationsQuery.size
    }, {
      allow_no_indices: successfulGenerationsQuery.allow_no_indices,
      index: eventLogIndex,
      ignore_unavailable: successfulGenerationsQuery.ignore_unavailable
    }, {
      aggs: successfulGenerationsQuery.aggs,
      query: successfulGenerationsQuery.query,
      size: successfulGenerationsQuery.size
    }],
    ignore_unavailable: true
  };
  const msearchResults = await esClient.msearch(mSearchQueryBody);
  const generationsResponse = msearchResults.responses[0];
  const successfulGenerationsResponse = msearchResults.responses[1];

  // transform the generations response:
  const rawGenerationsResponse = {
    aggregations: (0, _fp.get)('aggregations', generationsResponse)
  };
  const transformedGenerations = (0, _transform_get_attack_discovery_generations_search_result.transformGetAttackDiscoveryGenerationsSearchResult)({
    logger,
    rawResponse: rawGenerationsResponse
  });

  // transform the successful generations response:
  const rawSuccessfulGenerationsResponse = {
    aggregations: (0, _fp.get)('aggregations', successfulGenerationsResponse)
  };
  const successfulGenerationsMetadata = (0, _transform_successful_generations_search_result.transformSuccessfulGenerationsSearchResult)({
    logger,
    rawResponse: rawSuccessfulGenerationsResponse
  });

  // Combine the transformed generations with the successful generations metadata:
  return (0, _combine_generations_with_success_metadata.combineGenerationsWithSuccessMetadata)({
    transformedGenerations,
    successfulGenerationsMetadata
  });
};
exports.getAttackDiscoveryGenerations = getAttackDiscoveryGenerations;