"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getPerformanceTrackersGroupedById = exports.getPerformanceTrackersByType = exports.getMeanFromPerformanceMeasures = exports.findMarkerByNamePostfix = exports.createPerformanceTracker = exports.clearPerformanceTrackersByType = exports.PERFORMANCE_TRACKER_TYPES = exports.PERFORMANCE_TRACKER_MEASURES = exports.PERFORMANCE_TRACKER_MARKS = void 0;
var _uuid = require("uuid");
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * PERFORMANCE_TRACKER_TYPES defines top-level categories to be used as
 * the mark name. They are used to group marks and measures by type.
 */
const PERFORMANCE_TRACKER_TYPES = exports.PERFORMANCE_TRACKER_TYPES = {
  PANEL: 'Panel'
};
/**
 * PerformanceTrackerMarks are the marks that can be used to track performance
 * of lens charts. They are used to mark specific points in time during
 * the chart's lifecycle.
 */
const PERFORMANCE_TRACKER_MARKS = exports.PERFORMANCE_TRACKER_MARKS = {
  /**
   * Mark that indicates the start of everything before the rendering process.
   */
  PRE_RENDER: 'preRender',
  /**
   * Mark that indicates the start of the rendering process.
   * Should be used right before returning the chart's JSX.
   */
  RENDER_START: 'renderStart',
  /**
   * Mark that indicates the end of the rendering process.
   * Should be used at the beginning of the `renderComplete` callback.
   */
  RENDER_COMPLETE: 'renderComplete'
};
const PERFORMANCE_TRACKER_MEASURES = exports.PERFORMANCE_TRACKER_MEASURES = {
  PRE_RENDER_DURATION: 'preRenderDuration',
  RENDER_DURATION: 'renderDuration'
};

/**
 * Options to create a performance tracker.
 */

/**
 * Creates a performance tracker to mark and measure performance events.
 * @param options.type - High-level type of the performance tracker, for example "Panel".
 * @param options.subType - Lower-level type of the performance tracker type, for example "xyVis".
 * @returns A performance tracker object with a mark method.
 */
const createPerformanceTracker = ({
  type,
  subType,
  logger
}) => {
  const id = (0, _uuid.v4)();
  const createMarkName = name => `${type}:${subType}:${name}`;
  return {
    /**
     * Creates a performance mark with the given name.
     * @param name - The name of the mark to create, will be postfixed.
     * @returns The created performance mark
     */
    mark: name => {
      try {
        performance.mark(createMarkName(name), {
          detail: {
            id
          }
        });
      } catch (error) {
        logger === null || logger === void 0 ? void 0 : logger.error('Error creating performance mark:', error);
      }
    }
  };
};

/**
 * Finds a performance marker by its name postfix.
 * @param markers
 * @param namePostfix
 * @returns The found performance marker or undefined if not found.
 */
exports.createPerformanceTracker = createPerformanceTracker;
const findMarkerByNamePostfix = (markers, namePostfix) => markers.find(marker => marker.name.endsWith(`:${namePostfix}`));

/**
 * Get all performance trackers by type.
 * @param type - High-level type of the performance tracker, for example "Panel".
 * @returns An array of performance trackers.
 */
exports.findMarkerByNamePostfix = findMarkerByNamePostfix;
const getPerformanceTrackersByType = type => {
  try {
    return performance.getEntriesByType('mark').filter(marker => marker.name.startsWith(`${type}:`));
  } catch (e) {
    // Fail silently if performance API is not supported.
    return [];
  }
};

/**
 * Get all performance trackers grouped by id.
 * @param type - High-level type of the performance tracker, for example "Panel".
 * @returns A map of performance trackers grouped by id.
 */
exports.getPerformanceTrackersByType = getPerformanceTrackersByType;
const getPerformanceTrackersGroupedById = type => {
  try {
    return (0, _lodash.groupBy)(getPerformanceTrackersByType(type), marker => {
      var _marker$detail;
      return (_marker$detail = marker.detail) === null || _marker$detail === void 0 ? void 0 : _marker$detail.id;
    });
  } catch (e) {
    // Fail silently if performance API is not supported.
    return {};
  }
};

/**
 * Clear all performance trackers by type.
 * @param type - High-level type of the performance tracker, for example "Panel".
 */
exports.getPerformanceTrackersGroupedById = getPerformanceTrackersGroupedById;
const clearPerformanceTrackersByType = type => {
  try {
    getPerformanceTrackersByType(type).forEach(marker => performance.clearMarks(marker.name));
  } catch (e) {
    // Fail silently if performance API is not supported.
  }
};
exports.clearPerformanceTrackersByType = clearPerformanceTrackersByType;
/**
 * Get the mean duration of performance measures between two marks.
 * @param type
 * @param startMark
 * @param endMark
 * @param createPerformanceMeasures - Whether to create performance measures.
 * @returns The mean duration of the performance measures between the two marks.
 */
const getMeanFromPerformanceMeasures = ({
  type,
  startMark,
  endMark,
  createPerformanceMeasures = true
}) => {
  const groupedMarkers = getPerformanceTrackersGroupedById(type);
  if (Object.keys(groupedMarkers).length === 0) {
    return 0;
  }

  // `groupedMarkers` is a map of performance markers grouped by id.
  // Each group contains the performance markers for a single panel.
  // We need to extract the start and end times of the preRender, renderStart
  // and renderComplete markers and calculate the duration of each phase.
  const measurements = Object.values(groupedMarkers).map(markers => {
    var _findMarkerByNamePost, _findMarkerByNamePost2;
    const markerName = Array.isArray(markers) && markers.length > 0 ? markers[0].name.split(':').slice(0, -1).join(':') : undefined;
    const startTime = (_findMarkerByNamePost = findMarkerByNamePostfix(markers, startMark)) === null || _findMarkerByNamePost === void 0 ? void 0 : _findMarkerByNamePost.startTime;
    const endTime = (_findMarkerByNamePost2 = findMarkerByNamePostfix(markers, endMark)) === null || _findMarkerByNamePost2 === void 0 ? void 0 : _findMarkerByNamePost2.startTime;
    if (createPerformanceMeasures && markerName && startTime && endTime) {
      performance.measure(`${markerName}:${PERFORMANCE_TRACKER_MEASURES.PRE_RENDER_DURATION}`, {
        start: startTime,
        end: endTime
      });
    }
    return startTime && endTime ? endTime - startTime : 0;
  });
  return (0, _lodash.round)((0, _lodash.mean)(measurements), 2);
};
exports.getMeanFromPerformanceMeasures = getMeanFromPerformanceMeasures;