"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getFieldInfo = void 0;
exports.getIndexPatterns = getIndexPatterns;
exports.isRollup = exports.getTags = void 0;
var _public = require("@kbn/data-views-plugin/public");
var _i18n = require("@kbn/i18n");
var _rollup = require("@kbn/rollup");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const defaultIndexPatternListName = _i18n.i18n.translate('indexPatternManagement.editIndexPattern.list.defaultIndexPatternListName', {
  defaultMessage: 'Default'
});
const isRollup = (indexPatternType = '') => {
  return indexPatternType === _public.DataViewType.ROLLUP;
};
exports.isRollup = isRollup;
async function getIndexPatterns(defaultIndex, dataViewsService) {
  const existingIndexPatterns = await dataViewsService.getIdsWithTitle(true);
  const indexPatternsListItems = existingIndexPatterns.map(idxPattern => {
    const {
      id,
      title,
      namespaces,
      name
    } = idxPattern;
    const isDefault = defaultIndex === id;
    const tags = getTags(idxPattern, isDefault, dataViewsService.getRollupsEnabled());
    const displayName = name ? name : title;
    return {
      id,
      namespaces,
      title,
      name,
      default: isDefault,
      tags,
      // the prepending of 0 at the default pattern takes care of prioritization
      // so the sorting will but the default index on top
      // or on bottom of the table
      sort: `${isDefault ? '0' : '1'}${displayName}`,
      getName: () => displayName
    };
  });
  return indexPatternsListItems.sort((a, b) => {
    if (a.sort < b.sort) {
      return -1;
    } else if (a.sort > b.sort) {
      return 1;
    } else {
      return 0;
    }
  }) || [];
}
const getTags = (indexPattern, isDefault, rollupsEnabled) => {
  const tags = [];
  if (isDefault) {
    tags.push({
      key: _public.DataViewType.DEFAULT,
      name: defaultIndexPatternListName,
      'data-test-subj': 'default-tag'
    });
  }
  if (isRollup(indexPattern.type) && rollupsEnabled) {
    tags.push({
      key: _public.DataViewType.ROLLUP,
      name: _rollup.ROLLUP_DEPRECATION_BADGE_LABEL,
      'data-test-subj': 'rollup-tag'
    });
  }
  return tags;
};
exports.getTags = getTags;
const getFieldInfo = (indexPattern, field) => {
  var _indexPattern$typeMet;
  if (!isRollup(indexPattern.type)) {
    return [];
  }
  const allAggs = (_indexPattern$typeMet = indexPattern.typeMeta) === null || _indexPattern$typeMet === void 0 ? void 0 : _indexPattern$typeMet.aggs;
  const fieldAggs = allAggs && Object.keys(allAggs).filter(agg => allAggs[agg][field.name]);
  if (!fieldAggs || !fieldAggs.length) {
    return [];
  }
  return ['Rollup aggregations:'].concat(fieldAggs.map(aggName => {
    const agg = allAggs[aggName][field.name];
    switch (aggName) {
      case 'date_histogram':
        return _i18n.i18n.translate('indexPatternManagement.editIndexPattern.list.dateHistogramSummary', {
          defaultMessage: '{aggName} (interval: {interval}, {delay} {time_zone})',
          values: {
            aggName,
            interval: agg.fixed_interval,
            delay: agg.delay ? _i18n.i18n.translate('indexPatternManagement.editIndexPattern.list.DateHistogramDelaySummary', {
              defaultMessage: 'delay: {delay},',
              values: {
                delay: agg.delay
              }
            }) : '',
            time_zone: agg.time_zone
          }
        });
      case 'histogram':
        return _i18n.i18n.translate('indexPatternManagement.editIndexPattern.list.histogramSummary', {
          defaultMessage: '{aggName} (interval: {interval})',
          values: {
            aggName,
            interval: agg.interval
          }
        });
      default:
        return aggName;
    }
  }));
};
exports.getFieldInfo = getFieldInfo;