"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AnalyticsService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _apmRum = require("@elastic/apm-rum");
var _client = require("@elastic/ebt/client");
var _ebtTools = require("@kbn/ebt-tools");
var _track_performance_measure_entries = require("./track_performance_measure_entries");
var _track_clicks = require("./track_clicks");
var _get_session_id = require("./get_session_id");
var _track_viewport_size = require("./track_viewport_size");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/** @internal */

class AnalyticsService {
  constructor(core) {
    (0, _defineProperty2.default)(this, "analyticsClient", void 0);
    (0, _defineProperty2.default)(this, "subscriptionsHandler", new _rxjs.Subscription());
    this.analyticsClient = (0, _client.createAnalytics)({
      isDev: core.env.mode.dev,
      logger: core.logger.get('analytics'),
      getTraceContext: () => {
        var _apm$getCurrentTransa;
        return {
          id: (_apm$getCurrentTransa = _apmRum.apm.getCurrentTransaction()) === null || _apm$getCurrentTransa === void 0 ? void 0 : _apm$getCurrentTransa.traceId
        };
      }
    });
    this.registerBuildInfoAnalyticsContext(core);
    (0, _ebtTools.registerPerformanceMetricEventType)(this.analyticsClient);

    // We may eventually move the following to the client's package since they are not Kibana-specific
    // and can benefit other consumers of the client.
    this.registerSessionIdContext();
    this.registerBrowserInfoAnalyticsContext();
    this.subscriptionsHandler.add((0, _track_clicks.trackClicks)(this.analyticsClient, core.env.mode.dev));
    this.subscriptionsHandler.add((0, _track_performance_measure_entries.trackPerformanceMeasureEntries)(this.analyticsClient, core.env.mode.dev));
    this.subscriptionsHandler.add((0, _track_viewport_size.trackViewportSize)(this.analyticsClient));

    // Register a flush method in the browser so CI can explicitly call it before closing the browser.
    window.__kbnAnalytics = {
      flush: () => this.analyticsClient.flush()
    };
  }
  setup({
    injectedMetadata
  }) {
    this.registerElasticsearchInfoContext(injectedMetadata);
    return {
      optIn: this.analyticsClient.optIn,
      registerContextProvider: this.analyticsClient.registerContextProvider,
      removeContextProvider: this.analyticsClient.removeContextProvider,
      registerEventType: this.analyticsClient.registerEventType,
      registerShipper: this.analyticsClient.registerShipper,
      reportEvent: this.analyticsClient.reportEvent,
      telemetryCounter$: this.analyticsClient.telemetryCounter$
    };
  }
  start() {
    return {
      optIn: this.analyticsClient.optIn,
      reportEvent: this.analyticsClient.reportEvent,
      telemetryCounter$: this.analyticsClient.telemetryCounter$
    };
  }
  async stop() {
    this.subscriptionsHandler.unsubscribe();
    await this.analyticsClient.shutdown();
  }

  /**
   * Enriches the events with a session_id, so we can correlate them and understand funnels.
   * @internal
   */
  registerSessionIdContext() {
    this.analyticsClient.registerContextProvider({
      name: 'session-id',
      context$: (0, _rxjs.of)({
        session_id: (0, _get_session_id.getSessionId)()
      }),
      schema: {
        session_id: {
          type: 'keyword',
          _meta: {
            description: 'Unique session ID for every browser session'
          }
        }
      }
    });
  }

  /**
   * Enriches the event with the build information.
   * @param core The core context.
   * @internal
   */
  registerBuildInfoAnalyticsContext(core) {
    this.analyticsClient.registerContextProvider({
      name: 'build info',
      context$: (0, _rxjs.of)({
        isDev: core.env.mode.dev,
        isDistributable: core.env.packageInfo.dist,
        version: core.env.packageInfo.version,
        branch: core.env.packageInfo.branch,
        buildNum: core.env.packageInfo.buildNum,
        buildSha: core.env.packageInfo.buildSha
      }),
      schema: {
        isDev: {
          type: 'boolean',
          _meta: {
            description: 'Is it running in development mode?'
          }
        },
        isDistributable: {
          type: 'boolean',
          _meta: {
            description: 'Is it running from a distributable?'
          }
        },
        version: {
          type: 'keyword',
          _meta: {
            description: 'Version of the Kibana instance.'
          }
        },
        branch: {
          type: 'keyword',
          _meta: {
            description: 'Branch of source running Kibana from.'
          }
        },
        buildNum: {
          type: 'long',
          _meta: {
            description: 'Build number of the Kibana instance.'
          }
        },
        buildSha: {
          type: 'keyword',
          _meta: {
            description: 'Build SHA of the Kibana instance.'
          }
        }
      }
    });
  }

  /**
   * Enriches events with the current Browser's information
   * @internal
   */
  registerBrowserInfoAnalyticsContext() {
    this.analyticsClient.registerContextProvider({
      name: 'browser info',
      context$: (0, _rxjs.of)({
        user_agent: navigator.userAgent,
        preferred_language: navigator.language,
        preferred_languages: navigator.languages
      }),
      schema: {
        user_agent: {
          type: 'keyword',
          _meta: {
            description: 'User agent of the browser.'
          }
        },
        preferred_language: {
          type: 'keyword',
          _meta: {
            description: 'Preferred language of the browser.'
          }
        },
        preferred_languages: {
          type: 'array',
          items: {
            type: 'keyword',
            _meta: {
              description: 'List of the preferred languages of the browser.'
            }
          }
        }
      }
    });
  }

  /**
   * Enriches the events with the Elasticsearch info (cluster name, uuid and version).
   * @param injectedMetadata The injected metadata service.
   * @internal
   */
  registerElasticsearchInfoContext(injectedMetadata) {
    this.analyticsClient.registerContextProvider({
      name: 'elasticsearch info',
      context$: (0, _rxjs.of)(injectedMetadata.getElasticsearchInfo()),
      schema: {
        cluster_name: {
          type: 'keyword',
          _meta: {
            description: 'The Cluster Name',
            optional: true
          }
        },
        cluster_uuid: {
          type: 'keyword',
          _meta: {
            description: 'The Cluster UUID',
            optional: true
          }
        },
        cluster_version: {
          type: 'keyword',
          _meta: {
            description: 'The Cluster version',
            optional: true
          }
        },
        cluster_build_flavor: {
          type: 'keyword',
          _meta: {
            description: 'The Cluster build flavor',
            optional: true
          }
        }
      }
    });
  }
}
exports.AnalyticsService = AnalyticsService;