"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.unsafeTransformChangeToTransformFn = exports.dataRemovalChangeToTransformFn = exports.dataBackfillChangeToTransformFn = exports.buildModelVersionTransformFn = void 0;
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Build the transform function  for given model version, by chaining the transformations from its model changes.
 */
const buildModelVersionTransformFn = modelChanges => {
  const transformFns = [];
  modelChanges.forEach(change => {
    if (change.type === 'data_backfill') {
      transformFns.push(dataBackfillChangeToTransformFn(change));
    }
    if (change.type === 'data_removal') {
      transformFns.push(dataRemovalChangeToTransformFn(change));
    }
    if (change.type === 'unsafe_transform') {
      transformFns.push(unsafeTransformChangeToTransformFn(change));
    }
  });
  return mergeTransformFunctions(transformFns);
};
exports.buildModelVersionTransformFn = buildModelVersionTransformFn;
const dataRemovalChangeToTransformFn = change => {
  return (document, context) => {
    const attributes = document.attributes;
    change.removedAttributePaths.forEach(path => {
      (0, _lodash.unset)(attributes, path);
    });
    return {
      document
    };
  };
};
exports.dataRemovalChangeToTransformFn = dataRemovalChangeToTransformFn;
const dataBackfillChangeToTransformFn = change => {
  return (document, context) => {
    var _result$attributes;
    const result = change.backfillFn(document, context);
    const mergedAttributes = (0, _lodash.merge)({}, document.attributes, (_result$attributes = result.attributes) !== null && _result$attributes !== void 0 ? _result$attributes : {});
    return {
      document: {
        ...document,
        attributes: mergedAttributes
      }
    };
  };
};
exports.dataBackfillChangeToTransformFn = dataBackfillChangeToTransformFn;
const unsafeTransformChangeToTransformFn = change => {
  return change.transformFn;
};
exports.unsafeTransformChangeToTransformFn = unsafeTransformChangeToTransformFn;
const mergeTransformFunctions = transformFns => {
  if (transformFns.length === 0) {
    return noopTransform;
  }
  if (transformFns.length === 1) {
    return transformFns[0];
  }
  let mergedFn = transformFns[0];
  for (let i = 1; i < transformFns.length; i++) {
    mergedFn = merge(transformFns[i], mergedFn);
  }
  return mergedFn;
};
const noopTransform = doc => ({
  document: doc
});
const merge = (outer, inner) => {
  return (document, context) => {
    return outer(inner(document, context).document, context);
  };
};