"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MetadataService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _remove_undefined = require("./remove_undefined");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class MetadataService {
  constructor(config, logger) {
    (0, _defineProperty2.default)(this, "_userMetadata$", new _rxjs.BehaviorSubject(undefined));
    (0, _defineProperty2.default)(this, "stop$", new _rxjs.Subject());
    this.config = config;
    this.logger = logger;
  }
  setup(initialUserMetadata) {
    this._userMetadata$.next(initialUserMetadata);

    // Calculate `inTrial` based on the `trialEndDate`.
    // Elastic Cloud allows customers to end their trials earlier or even extend it in some cases, but this is a good compromise for now.
    const trialEndDate = initialUserMetadata.trial_end_date;
    if (trialEndDate) {
      this.scheduleUntil(() => ({
        in_trial: Date.now() <= new Date(trialEndDate).getTime()
      }),
      // Stop recalculating inTrial when the user is no-longer in trial
      metadata => metadata.in_trial === false);
    }
  }
  get userMetadata$() {
    return this._userMetadata$.pipe((0, _rxjs.filter)(Boolean),
    // Ensure we don't return undefined
    (0, _rxjs.debounceTime)(100),
    // Swallows multiple emissions that may occur during bootstrap
    (0, _rxjs.distinct)(meta => [meta.in_trial, meta.has_data].join('-')),
    // Checks if any of the dynamic fields have changed
    (0, _rxjs.map)(metadata => {
      const context = {
        kind: 'multi',
        ...(metadata.instanceKey && {
          kibana: (0, _remove_undefined.removeUndefined)({
            key: metadata.instanceKey,
            offering: metadata.offering,
            version: metadata.version,
            build_num: metadata.build_num,
            build_sha: metadata.build_sha,
            build_sha_short: metadata.build_sha_short,
            project_type: metadata.project_type,
            orchestrator_target: metadata.orchestrator_target,
            has_data: metadata.has_data
          })
        }),
        ...(metadata.organizationKey && {
          organization: (0, _remove_undefined.removeUndefined)({
            key: metadata.organizationKey,
            is_elastic_staff: metadata.is_elastic_staff,
            in_trial: metadata.in_trial,
            trial_end_date: metadata.trial_end_date
          })
        })
      };
      return context;
    }), (0, _rxjs.shareReplay)(1));
  }
  start({
    hasDataFetcher
  }) {
    // If no initial metadata (setup was not called) => it should not schedule any metadata extension
    if (!this._userMetadata$.value) return;
    this.scheduleUntil(async () => hasDataFetcher(),
    // Stop checking the moment the user has any data
    metadata => metadata.has_data === true);
  }
  stop() {
    this.stop$.next();
    this._userMetadata$.complete();
  }

  /**
   * Schedules a timer that calls `fn` to update the {@link FlatMetadata} until `untilFn` returns true.
   * @param fn Method to calculate the dynamic metadata.
   * @param untilFn Method that returns true when the scheduler should stop calling fn (potentially because the dynamic value is not expected to change anymore).
   * @internal
   */
  scheduleUntil(fn, untilFn) {
    (0, _rxjs.timer)(0, this.config.metadata_refresh_interval.asMilliseconds()).pipe((0, _rxjs.takeUntil)(this.stop$), (0, _rxjs.exhaustMap)(async () => {
      try {
        this._userMetadata$.next({
          ...this._userMetadata$.value,
          // We are running the schedules after the initial user metadata is set
          ...(await fn())
        });
      } catch (err) {
        this.logger.warn(`Failed to update metadata because ${err}`);
      }
    }), (0, _rxjs.takeWhile)(() => {
      return !untilFn(this._userMetadata$.value);
    })).subscribe();
  }
}
exports.MetadataService = MetadataService;