"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const server_sdk_1 = require("@openfeature/server-sdk");
/**
 * Convert an `errorKind` into an OpenFeature `errorCode`.
 * @param {string} errorKind The error kind to translate.
 * @returns {string} The OpenFeature error code.
 */
function translateErrorKind(errorKind) {
    // Error code specification.
    // https://github.com/open-feature/spec/blob/main/specification/sections/02-providers.md#requirement-28
    switch (errorKind) {
        case 'CLIENT_NOT_READY':
            return server_sdk_1.ErrorCode.PROVIDER_NOT_READY;
        case 'MALFORMED_FLAG':
            return server_sdk_1.ErrorCode.PARSE_ERROR;
        case 'FLAG_NOT_FOUND':
            return server_sdk_1.ErrorCode.FLAG_NOT_FOUND;
        case 'USER_NOT_SPECIFIED':
            return server_sdk_1.ErrorCode.TARGETING_KEY_MISSING;
        // General errors.
        default:
            return server_sdk_1.ErrorCode.GENERAL;
    }
}
/**
 * Translate an {@link LDEvaluationDetail} to a {@link ResolutionDetails}.
 * @param result The {@link LDEvaluationDetail} to translate.
 * @returns An equivalent {@link ResolutionDetails}.
 *
 * @internal
 */
function translateResult(result) {
    const resolution = {
        value: result.value,
        variant: result.variationIndex?.toString(),
        reason: result.reason.kind,
    };
    if (result.reason.kind === 'ERROR') {
        resolution.errorCode = translateErrorKind(result.reason.errorKind);
    }
    return resolution;
}
exports.default = translateResult;
//# sourceMappingURL=translateResult.js.map