"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.allAsync = exports.firstSeriesAsync = exports.allSeriesAsync = exports.firstResult = void 0;
/**
 * Iterate a collection any apply the specified operation. The first operation which
 * returns a value will be returned and iteration will stop.
 *
 * @param collection The collection to enumerate.
 * @param operator The operation to apply to each item.
 * @returns The result of the first successful operation.
 */
function firstResult(collection, operator) {
    let res;
    collection === null || collection === void 0 ? void 0 : collection.some((item, index) => {
        res = operator(item, index);
        return !!res;
    });
    return res;
}
exports.firstResult = firstResult;
const ITERATION_RECURSION_LIMIT = 50;
function seriesAsync(collection, check, all, index, cb) {
    if (!collection) {
        cb(false);
        return;
    }
    if (index < (collection === null || collection === void 0 ? void 0 : collection.length)) {
        check(collection[index], index, (res) => {
            if (all) {
                if (!res) {
                    cb(false);
                    return;
                }
            }
            else if (res) {
                cb(true);
                return;
            }
            if (collection.length > ITERATION_RECURSION_LIMIT) {
                // When we hit the recursion limit we defer execution
                // by using a resolved promise. This is similar to using setImmediate
                // but more portable.
                Promise.resolve().then(() => {
                    seriesAsync(collection, check, all, index + 1, cb);
                });
            }
            else {
                seriesAsync(collection, check, all, index + 1, cb);
            }
        });
    }
    else {
        cb(all);
    }
}
/**
 * Iterate a collection in series awaiting each check operation.
 * @param collection The collection to iterate.
 * @param check The check to perform for each item in the container.
 * @param cb Called with true if all items pass the check.
 */
function allSeriesAsync(collection, check, cb) {
    seriesAsync(collection, check, true, 0, cb);
}
exports.allSeriesAsync = allSeriesAsync;
/**
 * Iterate a collection in series awaiting each check operation.
 * @param collection The collection to iterate.
 * @param check The check to perform for each item in the container.
 * @param cb called with true on the first item that passes the check. False
 * means no items passed the check.
 */
function firstSeriesAsync(collection, check, cb) {
    seriesAsync(collection, check, false, 0, cb);
}
exports.firstSeriesAsync = firstSeriesAsync;
/**
 * Iterate a collection and execute the the given check operation
 * for all items concurrently.
 * @param collection The collection to iterate.
 * @param check The check to run for each item.
 * @param cb Callback executed when all items have been checked. The callback
 * will be called with true if each item resulted in true, otherwise it will
 * be called with false.
 */
function allAsync(collection, check, cb) {
    if (!collection) {
        cb(false);
        return;
    }
    Promise.all(collection === null || collection === void 0 ? void 0 : collection.map((item) => new Promise((resolve) => {
        check(item, resolve);
    }))).then((results) => {
        cb(results.every((success) => success));
    });
}
exports.allAsync = allAsync;
//# sourceMappingURL=collection.js.map