"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const js_sdk_common_1 = require("@launchdarkly/js-sdk-common");
const store_1 = require("../store");
const VersionedDataKinds_1 = require("../store/VersionedDataKinds");
const { initMetadataFromHeaders } = js_sdk_common_1.internal;
/**
 * @internal
 */
class PollingProcessor {
    constructor(_requestor, _pollInterval, _featureStore, _logger, _initSuccessHandler = () => { }, _errorHandler) {
        this._requestor = _requestor;
        this._pollInterval = _pollInterval;
        this._featureStore = _featureStore;
        this._logger = _logger;
        this._initSuccessHandler = _initSuccessHandler;
        this._errorHandler = _errorHandler;
        this._stopped = false;
    }
    _poll() {
        var _a;
        if (this._stopped) {
            return;
        }
        const reportJsonError = (data) => {
            var _a, _b, _c;
            (_a = this._logger) === null || _a === void 0 ? void 0 : _a.error('Polling received invalid data');
            (_b = this._logger) === null || _b === void 0 ? void 0 : _b.debug(`Invalid JSON follows: ${data}`);
            (_c = this._errorHandler) === null || _c === void 0 ? void 0 : _c.call(this, new js_sdk_common_1.LDPollingError(js_sdk_common_1.DataSourceErrorKind.InvalidData, 'Malformed JSON data in polling response'));
        };
        const startTime = Date.now();
        (_a = this._logger) === null || _a === void 0 ? void 0 : _a.debug('Polling LaunchDarkly for feature flag updates');
        this._requestor.requestAllData((err, body, headers) => {
            var _a, _b, _c, _d;
            const elapsed = Date.now() - startTime;
            const sleepFor = Math.max(this._pollInterval * 1000 - elapsed, 0);
            (_a = this._logger) === null || _a === void 0 ? void 0 : _a.debug('Elapsed: %d ms, sleeping for %d ms', elapsed, sleepFor);
            if (err) {
                const { status } = err;
                if (status && !(0, js_sdk_common_1.isHttpRecoverable)(status)) {
                    const message = (0, js_sdk_common_1.httpErrorMessage)(err, 'polling request');
                    (_b = this._logger) === null || _b === void 0 ? void 0 : _b.error(message);
                    (_c = this._errorHandler) === null || _c === void 0 ? void 0 : _c.call(this, new js_sdk_common_1.LDPollingError(js_sdk_common_1.DataSourceErrorKind.ErrorResponse, message, status));
                    // It is not recoverable, return and do not trigger another
                    // poll.
                    return;
                }
                (_d = this._logger) === null || _d === void 0 ? void 0 : _d.warn((0, js_sdk_common_1.httpErrorMessage)(err, 'polling request', 'will retry'));
            }
            else if (body) {
                const parsed = (0, store_1.deserializePoll)(body);
                if (!parsed) {
                    // We could not parse this JSON. Report the problem and fallthrough to
                    // start another poll.
                    reportJsonError(body);
                }
                else {
                    const initData = {
                        [VersionedDataKinds_1.default.Features.namespace]: parsed.flags,
                        [VersionedDataKinds_1.default.Segments.namespace]: parsed.segments,
                    };
                    this._featureStore.init(initData, () => {
                        this._initSuccessHandler();
                        // Triggering the next poll after the init has completed.
                        this._timeoutHandle = setTimeout(() => {
                            this._poll();
                        }, sleepFor);
                    }, initMetadataFromHeaders(headers));
                    // The poll will be triggered by  the feature store initialization
                    // completing.
                    return;
                }
            }
            // Falling through, there was some type of error and we need to trigger
            // a new poll.
            this._timeoutHandle = setTimeout(() => {
                this._poll();
            }, sleepFor);
        });
    }
    start() {
        this._poll();
    }
    stop() {
        if (this._timeoutHandle) {
            clearTimeout(this._timeoutHandle);
            this._timeoutHandle = undefined;
        }
        this._stopped = true;
    }
    close() {
        this.stop();
    }
}
exports.default = PollingProcessor;
//# sourceMappingURL=PollingProcessor.js.map