"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Tooltip = Tooltip;
var _d = _interopRequireDefault(require("d3"));
var _lodash = _interopRequireDefault(require("lodash"));
var _jquery = _interopRequireDefault(require("jquery"));
var _binder = require("../../lib/binder");
var _position_tooltip = require("./position_tooltip");
var _uiTheme = require("@kbn/ui-theme");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

let allContents = [];
const tooltipColumnPadding = parseInt(_uiTheme.euiLightVars.euiSizeXS || 0, 10) * 2;
const tooltipTableMargin = parseInt(_uiTheme.euiLightVars.euiSizeS || 0, 10) * 2;
const tooltipMaxWidth = parseInt(_uiTheme.euiLightVars.euiSizeXL || 0, 10) * 10;

/**
 * Add tooltip and listeners to visualization elements
 *
 * @class Tooltip
 * @constructor
 * @param el {HTMLElement} Reference to DOM element
 * @param formatter {Function} Tooltip formatter
 * @param events {Constructor} Allows tooltip to return event response data
 */
function Tooltip(id, el, formatter, events, uiSettings) {
  if (!(this instanceof Tooltip)) {
    return new Tooltip(id, el, formatter, events, uiSettings);
  }
  this.id = id; // unique id for this tooltip type
  this.el = el;
  this.order = 100; // higher ordered contents are rendered below the others
  this.formatter = formatter;
  this.events = events;
  this.uiSettings = uiSettings;
  this.containerClass = 'visWrapper';
  this.tooltipClass = 'visTooltip';
  this.tooltipSizerClass = 'visTooltip__sizingClone';
  this.showCondition = _lodash.default.constant(true);
  this.binder = new _binder.Binder();
}

/**
 * Get jquery reference to the tooltip node
 *
 * @return {Object} jQuery node object
 */
Tooltip.prototype.$get = _lodash.default.once(function () {
  return (0, _jquery.default)('<div>').addClass(this.tooltipClass).appendTo(document.body);
});

/**
 * Get jquery reference to the tooltip sizer node
 *
 * @return {Object} jQuery node object
 */
Tooltip.prototype.$getSizer = _lodash.default.once(function () {
  return this.$get().clone().removeClass(this.tooltipClass).addClass(this.tooltipSizerClass).appendTo(document.body);
});

/**
 * Show the tooltip, positioning it based on the content and chart container
 */
Tooltip.prototype.show = function () {
  const $tooltip = this.$get();
  const $chart = this.$getChart();
  const html = $tooltip.html();
  if (!$chart) return;
  const placement = (0, _position_tooltip.positionTooltip)({
    $window: (0, _jquery.default)(window),
    $chart: $chart,
    $el: $tooltip,
    $sizer: this.$getSizer(),
    event: _d.default.event
  }, html);
  $tooltip.css({
    visibility: 'visible',
    left: placement.left,
    top: placement.top
  });
  // The number of columns on the tooltip is currently the only
  // thing that differenciate one tooltip; from another
  const tooltipColumns = $tooltip.find('tbody > tr:nth-of-type(1) > td').length;
  if (tooltipColumns === 2) {
    // on pointseries tooltip
    const tooltipWidth = $tooltip.outerWidth();
    // get the last column to the right
    const valueColumn = $tooltip.find('tr:nth-of-type(1) > td:nth-child(2)');
    if (valueColumn.length !== 1) {
      return;
    }
    const valueColumnSize = valueColumn.outerWidth();
    const isGratherThanHalf = valueColumnSize > tooltipWidth / 2;
    const containerMaxWidth = isGratherThanHalf ? tooltipWidth / 2 - tooltipTableMargin - tooltipColumnPadding * 2 : tooltipWidth - valueColumnSize - tooltipTableMargin - tooltipColumnPadding;
    $tooltip.find('.visTooltip__labelContainer').css({
      'max-width': containerMaxWidth
    });
    if (isGratherThanHalf && tooltipWidth === tooltipMaxWidth) {
      $tooltip.find('.visTooltip__valueContainer').css({
        'max-width': containerMaxWidth
      });
    }
  } else if (tooltipColumns === 3) {
    // on hierarchical tooltip
    const tooltipWidth = $tooltip.outerWidth();
    // get the last column to the right (3rd column)
    const valueColumn = $tooltip.find('tr:nth-of-type(1) > td:nth-child(3)');
    if (valueColumn.length !== 1) {
      return;
    }
    const valueColumnSize = valueColumn.outerWidth();
    const containerMaxWidth = (tooltipWidth - valueColumnSize - tooltipTableMargin) / 2 - tooltipColumnPadding;
    $tooltip.find('.visTooltip__labelContainer').css({
      'max-width': containerMaxWidth
    });
  }
};

/**
 * Hide the tooltip, clearing its contents
 */
Tooltip.prototype.hide = function () {
  const $tooltip = this.$get();
  allContents = [];
  $tooltip.css({
    visibility: 'hidden',
    left: '-500px',
    top: '-500px'
  });
};

/**
 * Get the jQuery chart node, based on the container object
 * NOTE: the container is a d3 selection
 *
 * @return {Object} jQuery node for the chart
 */
Tooltip.prototype.$getChart = function () {
  const chart = (0, _jquery.default)(this.container && this.container.node());
  return chart.length ? chart : false;
};

/**
 * Renders tooltip
 *
 * @method render
 * @return {Function} Renders tooltip on a D3 selection
 */
Tooltip.prototype.render = function () {
  const self = this;

  /**
   * Calculates values for the tooltip placement
   *
   * @param {Object} selection D3 selection object
   */
  return function (selection) {
    const $tooltip = self.$get();
    const id = self.id;
    const order = self.order;
    if (self.container === undefined || self.container !== _d.default.select(self.el).select('.' + self.containerClass)) {
      self.container = _d.default.select(self.el).select('.' + self.containerClass);
    }
    const $chart = self.$getChart();
    if ($chart) {
      self.binder.jqOn($chart, 'mouseleave', function () {
        // only clear when we leave the chart, so that
        // moving between points doesn't make it reposition
        $chart.removeData('previousPlacement');
      });
    }
    selection.each(function (d, i) {
      const element = _d.default.select(this);
      function render(html) {
        allContents = _lodash.default.filter(allContents, function (content) {
          return content.id !== id;
        });
        if (html) allContents.push({
          id,
          html,
          order
        });
        const allHtml = (0, _lodash.default)(allContents).sortBy('order').map('html').compact().join('\n');
        if (allHtml) {
          $tooltip.html(allHtml);
          self.show();
        } else {
          self.hide();
        }
      }
      self.binder.fakeD3Bind(this, 'mousemove', function () {
        if (!self.showCondition.call(element, d, i)) {
          return render();
        }
        const events = self.events ? self.events.eventResponse(d, i) : d;
        return render(self.formatter(events, self.uiSettings));
      });
      self.binder.fakeD3Bind(this, 'mouseleave', function () {
        render();
      });
    });
  };
};
Tooltip.prototype.destroy = function () {
  this.hide();
  this.binder.destroy();
};