"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getNodesWithLowDiskSpace = getNodesWithLowDiskSpace;
exports.registerNodeDiskSpaceRoute = registerNodeDiskSpaceRoute;
var _configSchema = require("@kbn/config-schema");
var _constants = require("../../common/constants");
var _es_version_precheck = require("../lib/es_version_precheck");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getNodesWithLowDiskSpace(nodeStats) {
  const nodeIds = Object.keys(nodeStats.nodes);
  const nodesWithLowDiskSpace = [];
  nodeIds.forEach(nodeId => {
    var _node$fs, _node$fs$data;
    const node = nodeStats.nodes[nodeId];
    node === null || node === void 0 ? void 0 : (_node$fs = node.fs) === null || _node$fs === void 0 ? void 0 : (_node$fs$data = _node$fs.data) === null || _node$fs$data === void 0 ? void 0 : _node$fs$data.forEach(dataPath => {
      // @ts-expect-error low_watermark_free_space_in_bytes is missing from the types
      const lowWatermark = dataPath.low_watermark_free_space_in_bytes;
      const bytesAvailable = dataPath.available_in_bytes;
      const fsWithLowDiskSpace = [];
      if (lowWatermark && bytesAvailable && bytesAvailable < lowWatermark) {
        fsWithLowDiskSpace.push({
          nodeId,
          nodeName: node.name || nodeId,
          available: new _configSchema.ByteSizeValue(bytesAvailable).toString()
        });
      }
      if (fsWithLowDiskSpace.length > 0) {
        // Having multiple data paths on a single node is deprecated in ES and considered rare
        // If multiple data paths are above the low watermark, pick the one with the lowest available space
        fsWithLowDiskSpace.sort((a, b) => {
          const aBytes = _configSchema.ByteSizeValue.parse(a.available).getValueInBytes();
          const bBytes = _configSchema.ByteSizeValue.parse(b.available).getValueInBytes();
          return aBytes - bBytes;
        });
        nodesWithLowDiskSpace.push(fsWithLowDiskSpace[0]);
      }
    });
  });
  return nodesWithLowDiskSpace;
}
function registerNodeDiskSpaceRoute({
  router,
  lib: {
    handleEsError
  }
}) {
  router.get({
    path: `${_constants.API_BASE_PATH}/node_disk_space`,
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on es client for authorization'
      }
    },
    validate: false
  }, (0, _es_version_precheck.versionCheckHandlerWrapper)(async ({
    core
  }, request, response) => {
    try {
      const {
        elasticsearch: {
          client
        }
      } = await core;
      const nodeStats = await client.asCurrentUser.nodes.stats({
        metric: 'fs'
      });
      const nodesWithLowDiskSpace = getNodesWithLowDiskSpace(nodeStats);
      return response.ok({
        body: nodesWithLowDiskSpace
      });
    } catch (error) {
      return handleEsError({
        error,
        response
      });
    }
  }));
}