"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.readOnlyExecute = readOnlyExecute;
var _lodash = require("lodash");
var _types = require("../../../../../../common/types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_BATCH_SIZE = 10;
async function* readOnlyExecute(dataStreamName, meta, api, batchSize = DEFAULT_BATCH_SIZE) {
  const {
    indicesRequiringUpgrade
  } = meta || {};
  const startTimeMs = +Date.now();
  if (!indicesRequiringUpgrade || !indicesRequiringUpgrade.length) {
    return {
      migrationOp: {
        status: _types.DataStreamMigrationStatus.completed,
        resolutionType: 'readonly',
        taskPercComplete: 1,
        progressDetails: {
          startTimeMs,
          successCount: 0,
          pendingCount: 0,
          inProgressCount: 0,
          errorsCount: 0
        }
      }
    };
  }
  let processedCount = 0;
  const batches = (0, _lodash.chunk)(indicesRequiringUpgrade, batchSize);
  try {
    for (const batch of batches) {
      const {
        error
      } = await api.markIndicesAsReadOnly(dataStreamName, batch);
      if (error) {
        throw error;
      }
      processedCount += batch.length;
      const status = processedCount >= indicesRequiringUpgrade.length ? _types.DataStreamMigrationStatus.completed : _types.DataStreamMigrationStatus.inProgress;
      const taskPercComplete = processedCount / indicesRequiringUpgrade.length;
      yield {
        migrationOp: {
          resolutionType: 'readonly',
          status,
          taskPercComplete,
          progressDetails: {
            startTimeMs,
            successCount: processedCount,
            pendingCount: indicesRequiringUpgrade.length - processedCount,
            inProgressCount: batch.length,
            errorsCount: 0
          }
        }
      };
    }
  } catch (error) {
    return {
      migrationOp: {
        resolutionType: 'readonly',
        status: _types.DataStreamMigrationStatus.failed,
        errorMessage: error.message || 'Unknown error occurred'
      }
    };
  }
  return {
    migrationOp: {
      resolutionType: 'readonly',
      status: _types.DataStreamMigrationStatus.completed,
      taskPercComplete: 1,
      progressDetails: {
        startTimeMs,
        successCount: indicesRequiringUpgrade.length,
        pendingCount: 0,
        inProgressCount: 0,
        errorsCount: 0
      }
    }
  };
}