"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useStateProps = void 0;
var _common = require("@kbn/data-views-plugin/common");
var _esQuery = require("@kbn/es-query");
var _esqlUtils = require("@kbn/esql-utils");
var _dataViewUtils = require("@kbn/data-view-utils");
var _react = require("react");
var _state_selectors = require("../utils/state_selectors");
var _use_state_selector = require("../utils/use_state_selector");
var _local_storage_utils = require("../utils/local_storage_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const useStateProps = ({
  services,
  localStorageKeyPrefix,
  stateService,
  dataView,
  query,
  searchSessionId,
  requestAdapter,
  columns,
  breakdownField,
  onBreakdownFieldChange: originalOnBreakdownFieldChange
}) => {
  const chartHidden = (0, _use_state_selector.useStateSelector)(stateService === null || stateService === void 0 ? void 0 : stateService.state$, _state_selectors.chartHiddenSelector);
  const timeInterval = (0, _use_state_selector.useStateSelector)(stateService === null || stateService === void 0 ? void 0 : stateService.state$, _state_selectors.timeIntervalSelector);
  const totalHitsResult = (0, _use_state_selector.useStateSelector)(stateService === null || stateService === void 0 ? void 0 : stateService.state$, _state_selectors.totalHitsResultSelector);
  const totalHitsStatus = (0, _use_state_selector.useStateSelector)(stateService === null || stateService === void 0 ? void 0 : stateService.state$, _state_selectors.totalHitsStatusSelector);
  const lensAdapters = (0, _use_state_selector.useStateSelector)(stateService === null || stateService === void 0 ? void 0 : stateService.state$, _state_selectors.lensAdaptersSelector);
  const lensDataLoading$ = (0, _use_state_selector.useStateSelector)(stateService === null || stateService === void 0 ? void 0 : stateService.state$, _state_selectors.lensDataLoadingSelector$);
  /**
   * Contexts
   */

  const isPlainRecord = (0, _react.useMemo)(() => {
    return query && (0, _esQuery.isOfAggregateQueryType)(query);
  }, [query]);
  const isTimeBased = (0, _react.useMemo)(() => {
    return dataView && dataView.type !== _common.DataViewType.ROLLUP && dataView.isTimeBased();
  }, [dataView]);
  const hits = (0, _react.useMemo)(() => {
    if (totalHitsResult instanceof Error) {
      return undefined;
    }
    return {
      status: totalHitsStatus,
      total: totalHitsResult
    };
  }, [totalHitsResult, totalHitsStatus]);
  const chart = (0, _react.useMemo)(() => {
    if (!isTimeBased && !isPlainRecord) {
      return undefined;
    }
    return {
      hidden: chartHidden,
      timeInterval
    };
  }, [chartHidden, isPlainRecord, isTimeBased, timeInterval]);
  const breakdown = (0, _react.useMemo)(() => {
    if (!isTimeBased) {
      return undefined;
    }

    // hide the breakdown field selector when the ES|QL query has a transformational command (STATS, KEEP etc)
    if (query && (0, _esQuery.isOfAggregateQueryType)(query) && (0, _esqlUtils.hasTransformationalCommand)(query.esql)) {
      return undefined;
    }
    if (isPlainRecord) {
      const breakdownColumn = columns === null || columns === void 0 ? void 0 : columns.find(column => column.name === breakdownField);
      const field = breakdownColumn ? new _common.DataViewField((0, _dataViewUtils.convertDatatableColumnToDataViewFieldSpec)(breakdownColumn)) : undefined;
      return {
        field
      };
    }
    return {
      field: breakdownField ? dataView === null || dataView === void 0 ? void 0 : dataView.getFieldByName(breakdownField) : undefined
    };
  }, [isTimeBased, query, isPlainRecord, breakdownField, dataView, columns]);
  const request = (0, _react.useMemo)(() => {
    return {
      searchSessionId,
      adapter: requestAdapter
    };
  }, [requestAdapter, searchSessionId]);

  /**
   * Callbacks
   */

  const onTopPanelHeightChange = (0, _react.useCallback)(topPanelHeight => {
    stateService === null || stateService === void 0 ? void 0 : stateService.setTopPanelHeight(topPanelHeight);
  }, [stateService]);
  const onTimeIntervalChange = (0, _react.useCallback)(newTimeInterval => {
    stateService === null || stateService === void 0 ? void 0 : stateService.setTimeInterval(newTimeInterval);
  }, [stateService]);
  const onTotalHitsChange = (0, _react.useCallback)((newTotalHitsStatus, newTotalHitsResult) => {
    stateService === null || stateService === void 0 ? void 0 : stateService.setTotalHits({
      totalHitsStatus: newTotalHitsStatus,
      totalHitsResult: newTotalHitsResult
    });
  }, [stateService]);
  const onChartHiddenChange = (0, _react.useCallback)(newChartHidden => {
    stateService === null || stateService === void 0 ? void 0 : stateService.setChartHidden(newChartHidden);
  }, [stateService]);
  const onChartLoad = (0, _react.useCallback)(event => {
    // We need to store the Lens request adapter in order to inspect its requests
    stateService === null || stateService === void 0 ? void 0 : stateService.setLensRequestAdapter(event.adapters.requests);
    stateService === null || stateService === void 0 ? void 0 : stateService.setLensAdapters(event.adapters);
    stateService === null || stateService === void 0 ? void 0 : stateService.setLensDataLoading$(event.dataLoading$);
  }, [stateService]);
  const onBreakdownFieldChange = (0, _react.useCallback)(newBreakdownField => {
    originalOnBreakdownFieldChange === null || originalOnBreakdownFieldChange === void 0 ? void 0 : originalOnBreakdownFieldChange(newBreakdownField === null || newBreakdownField === void 0 ? void 0 : newBreakdownField.name);
  }, [originalOnBreakdownFieldChange]);
  const onSuggestionContextChange = (0, _react.useCallback)(suggestionContext => {
    stateService === null || stateService === void 0 ? void 0 : stateService.setCurrentSuggestionContext(suggestionContext);
  }, [stateService]);

  /**
   * Effects
   */

  // Sync the breakdown field with local storage
  (0, _react.useEffect)(() => {
    if (localStorageKeyPrefix) {
      (0, _local_storage_utils.setBreakdownField)(services.storage, localStorageKeyPrefix, breakdownField);
    }
  }, [breakdownField, localStorageKeyPrefix, services.storage]);

  // Clear the Lens request adapter when the chart is hidden
  (0, _react.useEffect)(() => {
    if (chartHidden || !chart) {
      stateService === null || stateService === void 0 ? void 0 : stateService.setLensRequestAdapter(undefined);
    }
  }, [chart, chartHidden, stateService]);
  return {
    hits,
    chart,
    breakdown,
    request,
    isPlainRecord,
    lensAdapters,
    dataLoading$: lensDataLoading$,
    onTopPanelHeightChange,
    onTimeIntervalChange,
    onTotalHitsChange,
    onChartHiddenChange,
    onChartLoad,
    onBreakdownFieldChange,
    onSuggestionContextChange
  };
};
exports.useStateProps = useStateProps;