"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isPerRowAggregation = exports.isGroupAggregation = exports.isCountAggregation = exports.buildAggregation = exports.MAX_SOURCE_FIELDS_TO_COPY = exports.DEFAULT_GROUPS = exports.BUCKET_SELECTOR_PATH_NAME = exports.BUCKET_SELECTOR_FIELD = void 0;
var _date_range_info = require("./date_range_info");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const BUCKET_SELECTOR_PATH_NAME = exports.BUCKET_SELECTOR_PATH_NAME = 'compareValue';
const BUCKET_SELECTOR_FIELD = exports.BUCKET_SELECTOR_FIELD = `params.${BUCKET_SELECTOR_PATH_NAME}`;
const DEFAULT_GROUPS = exports.DEFAULT_GROUPS = 100;
const MAX_SOURCE_FIELDS_TO_COPY = exports.MAX_SOURCE_FIELDS_TO_COPY = 10;
const MAX_TOP_HITS_SIZE = 100;
const isCountAggregation = aggType => aggType === 'count';
exports.isCountAggregation = isCountAggregation;
const isGroupAggregation = termField => !!termField;
exports.isGroupAggregation = isGroupAggregation;
const isPerRowAggregation = groupBy => groupBy === 'row';
exports.isPerRowAggregation = isPerRowAggregation;
const buildAggregation = ({
  timeSeries,
  aggType,
  aggField,
  termField,
  termSize,
  sourceFieldsParams,
  condition,
  topHitsSize,
  loggerCb
}) => {
  var _aggContainer$aggs;
  const aggContainer = {
    aggs: {}
  };
  const isCountAgg = isCountAggregation(aggType);
  const isGroupAgg = isGroupAggregation(termField);
  const isMultiTerms = Array.isArray(termField);
  const isDateAgg = !!timeSeries;
  const includeConditionInQuery = !!condition;
  let dateRangeInfo = null;
  if (isDateAgg) {
    const {
      timeWindowSize,
      timeWindowUnit,
      dateStart,
      dateEnd,
      interval
    } = timeSeries;
    const window = `${timeWindowSize}${timeWindowUnit}`;
    dateRangeInfo = (0, _date_range_info.getDateRangeInfo)({
      dateStart,
      dateEnd,
      window,
      interval
    });
  }

  // Cap the maximum number of terms returned to the resultLimit if defined
  // Use resultLimit + 1 because we're using the bucket selector aggregation
  // to apply the threshold condition to the ES query. We don't seem to be
  // able to get the true cardinality from the bucket selector (i.e., get
  // the number of buckets that matched the selector condition without actually
  // retrieving the bucket data). By using resultLimit + 1, we can count the number
  // of buckets returned and if the value is greater than resultLimit, we know that
  // there is additional alert data that we're not returning.
  let terms = termSize || DEFAULT_GROUPS;
  terms = includeConditionInQuery && condition.resultLimit ? terms > condition.resultLimit ? condition.resultLimit + 1 : terms : terms;
  let aggParent = aggContainer;
  const getAggName = () => isDateAgg ? 'sortValueAgg' : 'metricAgg';

  // first, add a group aggregation, if requested
  if (isGroupAgg) {
    aggParent.aggs = {
      groupAgg: {
        ...(isMultiTerms ? {
          multi_terms: {
            terms: termField.map(field => ({
              field
            })),
            size: terms
          }
        } : {
          terms: {
            field: termField,
            size: terms
          }
        })
      },
      ...(includeConditionInQuery ? {
        groupAggCount: {
          stats_bucket: {
            buckets_path: 'groupAgg._count'
          }
        }
      } : {})
    };

    // if not count add an order
    if (!isCountAgg) {
      const sortOrder = aggType === 'min' ? 'asc' : 'desc';
      if (isMultiTerms && aggParent.aggs.groupAgg.multi_terms) {
        aggParent.aggs.groupAgg.multi_terms.order = {
          [getAggName()]: sortOrder
        };
      } else if (aggParent.aggs.groupAgg.terms) {
        aggParent.aggs.groupAgg.terms.order = {
          [getAggName()]: sortOrder
        };
      }
    } else if (includeConditionInQuery) {
      aggParent.aggs.groupAgg.aggs = {
        conditionSelector: {
          bucket_selector: {
            buckets_path: {
              [BUCKET_SELECTOR_PATH_NAME]: '_count'
            },
            script: condition.conditionScript
          }
        }
      };
    }
    aggParent = aggParent.aggs.groupAgg;
  }

  // add sourceField aggregations
  if (sourceFieldsParams && sourceFieldsParams.length > 0) {
    sourceFieldsParams.forEach(field => {
      aggParent.aggs = {
        ...aggParent.aggs,
        [field.label]: {
          terms: {
            field: field.searchPath,
            size: MAX_SOURCE_FIELDS_TO_COPY
          }
        }
      };
    });
  }

  // next, add the time window aggregation
  if (isDateAgg) {
    aggParent.aggs = {
      ...aggParent.aggs,
      dateAgg: {
        date_range: {
          field: timeSeries.timeField,
          format: 'strict_date_time',
          ranges: dateRangeInfo.dateRanges
        }
      }
    };
  }
  if (isGroupAgg && topHitsSize) {
    if (topHitsSize > MAX_TOP_HITS_SIZE) {
      topHitsSize = MAX_TOP_HITS_SIZE;
      if (loggerCb) loggerCb(`Top hits size is capped at ${MAX_TOP_HITS_SIZE}`);
    }
    aggParent.aggs = {
      ...aggParent.aggs,
      topHitsAgg: {
        top_hits: {
          size: topHitsSize
        }
      }
    };
  }

  // if not count, add a sorted value agg
  if (!isCountAgg) {
    aggParent.aggs = {
      ...aggParent.aggs,
      [getAggName()]: {
        [aggType]: {
          field: aggField
        }
      }
    };
    if (isGroupAgg && includeConditionInQuery) {
      aggParent.aggs.conditionSelector = {
        bucket_selector: {
          buckets_path: {
            [BUCKET_SELECTOR_PATH_NAME]: getAggName()
          },
          script: condition.conditionScript
        }
      };
    }
  }
  if (timeSeries && dateRangeInfo) {
    var _aggParent$aggs$dateA, _aggParent, _aggParent$aggs;
    aggParent = (_aggParent$aggs$dateA = (_aggParent = aggParent) === null || _aggParent === void 0 ? void 0 : (_aggParent$aggs = _aggParent.aggs) === null || _aggParent$aggs === void 0 ? void 0 : _aggParent$aggs.dateAgg) !== null && _aggParent$aggs$dateA !== void 0 ? _aggParent$aggs$dateA : {};

    // finally, the metric aggregation, if requested
    if (!isCountAgg) {
      aggParent.aggs = {
        metricAgg: {
          [aggType]: {
            field: aggField
          }
        }
      };
    }
  }
  return (_aggContainer$aggs = aggContainer.aggs) !== null && _aggContainer$aggs !== void 0 ? _aggContainer$aggs : {};
};
exports.buildAggregation = buildAggregation;