"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.setupCapabilities = exports.hasPrivilegeFactory = exports.getPrivilegesAndCapabilities = exports.extractMissingPrivileges = exports.TRANSFORM_PLUGIN_ID = void 0;
var _mlIsPopulatedObject = require("@kbn/ml-is-populated-object");
var _capabilities = require("../common/types/capabilities");
var _constants = require("../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TRANSFORM_PLUGIN_ID = exports.TRANSFORM_PLUGIN_ID = 'transform';
function isPrivileges(arg) {
  return (0, _mlIsPopulatedObject.isPopulatedObject)(arg, ['hasAllPrivileges', 'missingPrivileges']) && typeof arg.hasAllPrivileges === 'boolean' && typeof arg.missingPrivileges === 'object' && arg.missingPrivileges !== null;
}
const hasPrivilegeFactory = privileges => privilege => {
  const [section, requiredPrivilege] = privilege;
  if (isPrivileges(privileges) && !privileges.missingPrivileges[section]) {
    // if the section does not exist in our missingPrivileges, everything is OK
    return true;
  }
  if (isPrivileges(privileges) && privileges.missingPrivileges[section].length === 0) {
    return true;
  }
  if (requiredPrivilege === '*') {
    // If length > 0 and we require them all... KO
    return false;
  }
  // If we require _some_ privilege, we make sure that the one
  // we require is *not* in the missingPrivilege array
  return isPrivileges(privileges) && !privileges.missingPrivileges[section].includes(requiredPrivilege);
};
exports.hasPrivilegeFactory = hasPrivilegeFactory;
const extractMissingPrivileges = (privilegesObject = {}) => Object.keys(privilegesObject).reduce((privileges, privilegeName) => {
  if (!privilegesObject[privilegeName]) {
    privileges.push(privilegeName);
  }
  return privileges;
}, []);
exports.extractMissingPrivileges = extractMissingPrivileges;
const getPrivilegesAndCapabilities = (clusterPrivileges, hasOneIndexWithAllPrivileges, hasAllPrivileges) => {
  const privilegesResult = {
    hasAllPrivileges: true,
    missingPrivileges: {
      cluster: [],
      index: []
    }
  };

  // Find missing cluster privileges and set overall app privileges
  privilegesResult.missingPrivileges.cluster = extractMissingPrivileges(clusterPrivileges);
  privilegesResult.hasAllPrivileges = hasAllPrivileges;
  if (!hasOneIndexWithAllPrivileges) {
    privilegesResult.missingPrivileges.index = [..._constants.APP_INDEX_PRIVILEGES];
  }
  const hasPrivilege = hasPrivilegeFactory(privilegesResult);
  const capabilities = (0, _capabilities.getInitialTransformCapabilities)();
  capabilities.canGetTransform = hasPrivilege(['cluster', 'cluster:monitor/transform/get']) && hasPrivilege(['cluster', 'cluster:monitor/transform/stats/get']);
  capabilities.canCreateTransform = hasPrivilege(['cluster', 'cluster:admin/transform/put']);
  capabilities.canDeleteTransform = hasPrivilege(['cluster', 'cluster:admin/transform/delete']);
  capabilities.canResetTransform = hasPrivilege(['cluster', 'cluster:admin/transform/reset']);
  capabilities.canPreviewTransform = hasPrivilege(['cluster', 'cluster:admin/transform/preview']);
  capabilities.canStartStopTransform = hasPrivilege(['cluster', 'cluster:admin/transform/start']) && hasPrivilege(['cluster', 'cluster:admin/transform/start_task']) && hasPrivilege(['cluster', 'cluster:admin/transform/stop']);
  capabilities.canCreateTransformAlerts = capabilities.canCreateTransform;
  capabilities.canUseTransformAlerts = capabilities.canGetTransform;
  capabilities.canScheduleNowTransform = capabilities.canStartStopTransform;
  capabilities.canReauthorizeTransform = capabilities.canStartStopTransform;
  capabilities.canDeleteIndex = hasAllPrivileges;
  return {
    privileges: privilegesResult,
    capabilities
  };
};
exports.getPrivilegesAndCapabilities = getPrivilegesAndCapabilities;
const setupCapabilities = (core, securitySetup) => {
  core.capabilities.registerProvider(() => {
    return {
      transform: (0, _capabilities.getInitialTransformCapabilities)()
    };
  });
  core.capabilities.registerSwitcher(async (request, capabilities, useDefaultCapabilities) => {
    var _securitySetup$licens, _privileges$elasticse;
    if (useDefaultCapabilities) {
      return {};
    }
    const isSecurityPluginEnabled = (_securitySetup$licens = securitySetup === null || securitySetup === void 0 ? void 0 : securitySetup.license.isEnabled()) !== null && _securitySetup$licens !== void 0 ? _securitySetup$licens : false;
    const startServices = await core.getStartServices();
    const [, {
      security: securityStart
    }] = startServices;

    // If security is not enabled or not available, transform should have full permission
    if (!isSecurityPluginEnabled || !securityStart) {
      return {
        transform: (0, _capabilities.getInitialTransformCapabilities)(true)
      };
    }
    const checkPrivileges = securityStart.authz.checkPrivilegesDynamicallyWithRequest(request);
    const {
      hasAllRequested,
      privileges
    } = await checkPrivileges({
      elasticsearch: {
        cluster: _constants.APP_CLUSTER_PRIVILEGES,
        index: {}
      }
    });
    const clusterPrivileges = Array.isArray(privileges === null || privileges === void 0 ? void 0 : (_privileges$elasticse = privileges.elasticsearch) === null || _privileges$elasticse === void 0 ? void 0 : _privileges$elasticse.cluster) ? privileges.elasticsearch.cluster.reduce((acc, p) => {
      acc[p.privilege] = p.authorized;
      return acc;
    }, {}) : {};
    const hasOneIndexWithAllPrivileges = false;
    const transformCapabilities = getPrivilegesAndCapabilities(clusterPrivileges, hasOneIndexWithAllPrivileges, hasAllRequested).capabilities;
    return {
      transform: transformCapabilities
    };
  }, {
    capabilityPath: 'transform.*'
  });
};
exports.setupCapabilities = setupCapabilities;