"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.latestConfigMapper = void 0;
exports.useLatestFunctionConfig = useLatestFunctionConfig;
exports.validateLatestConfig = validateLatestConfig;
var _react = require("react");
var _i18n = require("@kbn/i18n");
var _mlAggUtils = require("@kbn/ml-agg-utils");
var _app_dependencies = require("../../../../../app_dependencies");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Latest function config mapper between API and UI
 */
const latestConfigMapper = exports.latestConfigMapper = {
  toAPIConfig(uiConfig) {
    var _uiConfig$unique_key, _uiConfig$sort;
    return {
      unique_key: (_uiConfig$unique_key = uiConfig.unique_key) !== null && _uiConfig$unique_key !== void 0 && _uiConfig$unique_key.length ? uiConfig.unique_key.map(v => v.value) : [],
      sort: ((_uiConfig$sort = uiConfig.sort) === null || _uiConfig$sort === void 0 ? void 0 : _uiConfig$sort.value) !== undefined ? uiConfig.sort.value : ''
    };
  },
  toUIConfig() {}
};

/**
 * Provides available options for unique_key and sort fields
 * @param dataView
 * @param aggConfigs
 * @param runtimeMappings
 */
function getOptions(dataView, aggConfigs, runtimeMappings) {
  const aggConfig = aggConfigs.aggs[0];
  const param = aggConfig.type.params.find(p => p.type === 'field');
  const filteredDataViewFields = param ? param.getAvailableFields(aggConfig)
  // runtimeMappings may already include runtime fields defined by the data view
  .filter(ip => ip.runtimeField === undefined) : [];
  const ignoreFieldNames = new Set(['_source', '_type', '_index', '_id', '_version', '_score', '_ignored']);
  const runtimeFieldsOptions = runtimeMappings ? Object.entries(runtimeMappings).map(([fieldName, fieldMapping]) => ({
    label: fieldName,
    value: fieldName,
    field: {
      id: fieldName,
      type: fieldMapping.type
    }
  })) : [];
  const uniqueKeyOptions = filteredDataViewFields.filter(v => !ignoreFieldNames.has(v.name) && !(0, _mlAggUtils.isCounterTimeSeriesMetric)(v)).map(v => {
    var _v$esTypes;
    return {
      label: v.displayName,
      value: v.name,
      field: {
        id: v.name,
        type: Array.isArray(v.esTypes) && ((_v$esTypes = v.esTypes) === null || _v$esTypes === void 0 ? void 0 : _v$esTypes.length) > 0 ? v.esTypes[0] : 'keyword'
      }
    };
  });
  const runtimeFieldsSortOptions = runtimeMappings ? Object.entries(runtimeMappings).filter(([fieldName, fieldMapping]) => fieldMapping.type === 'date').map(([fieldName, fieldMapping]) => ({
    label: fieldName,
    value: fieldName,
    field: {
      id: fieldName,
      type: fieldMapping.type
    }
  })) : [];
  const dataViewFieldsSortOptions = dataView.fields
  // The backend API for `latest` allows all field types for sort but the UI will be limited to `date`.
  .filter(v => !ignoreFieldNames.has(v.name) && v.sortable && v.type === 'date').map(v => {
    var _v$esTypes2;
    return {
      label: v.displayName,
      value: v.name,
      field: {
        id: v.name,
        type: Array.isArray(v.esTypes) && ((_v$esTypes2 = v.esTypes) === null || _v$esTypes2 === void 0 ? void 0 : _v$esTypes2.length) > 0 ? v.esTypes[0] : 'keyword'
      }
    };
  });
  const sortByLabel = (a, b) => a.label.localeCompare(b.label);
  return {
    uniqueKeyOptions: [...uniqueKeyOptions, ...runtimeFieldsOptions].sort(sortByLabel),
    sortFieldOptions: [...dataViewFieldsSortOptions, ...runtimeFieldsSortOptions].sort(sortByLabel)
  };
}

/**
 * Validates latest function configuration
 */
function validateLatestConfig(config) {
  var _config$unique_key;
  const isValid = !!(config !== null && config !== void 0 && (_config$unique_key = config.unique_key) !== null && _config$unique_key !== void 0 && _config$unique_key.length) && typeof (config === null || config === void 0 ? void 0 : config.sort) === 'string' && (config === null || config === void 0 ? void 0 : config.sort.length) > 0;
  return {
    isValid,
    ...(isValid ? {} : {
      errorMessage: _i18n.i18n.translate('xpack.transform.latestPreview.latestPreviewIncompleteConfigCalloutBody', {
        defaultMessage: 'Please choose at least one unique key and sort field.'
      })
    })
  };
}
function useLatestFunctionConfig(defaults, dataView, runtimeMappings) {
  const [config, setLatestFunctionConfig] = (0, _react.useState)({
    unique_key: defaults.unique_key,
    sort: defaults.sort
  });
  const {
    data
  } = (0, _app_dependencies.useAppDependencies)();
  const {
    uniqueKeyOptions,
    sortFieldOptions
  } = (0, _react.useMemo)(() => {
    const aggConfigs = data.search.aggs.createAggConfigs(dataView, [{
      type: 'terms'
    }]);
    return getOptions(dataView, aggConfigs, runtimeMappings);
  }, [dataView, data.search.aggs, runtimeMappings]);
  const updateLatestFunctionConfig = (0, _react.useCallback)(update => setLatestFunctionConfig({
    ...config,
    ...update
  }), [config]);
  const requestPayload = (0, _react.useMemo)(() => {
    const latest = latestConfigMapper.toAPIConfig(config);
    return latest ? {
      latest
    } : undefined;
  }, [config]);
  const validationStatus = (0, _react.useMemo)(() => validateLatestConfig(requestPayload === null || requestPayload === void 0 ? void 0 : requestPayload.latest), [requestPayload === null || requestPayload === void 0 ? void 0 : requestPayload.latest]);
  return {
    config,
    uniqueKeyOptions,
    sortFieldOptions,
    updateLatestFunctionConfig,
    validationStatus,
    requestPayload
  };
}