"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DeployPrivateLocationMonitors = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _constants = require("@kbn/spaces-plugin/common/constants");
var _utils = require("../synthetics_service/utils");
var _monitor_config_repository = require("../services/monitor_config_repository");
var _format_configs = require("../synthetics_service/formatters/public_formatters/format_configs");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class DeployPrivateLocationMonitors {
  constructor(serverSetup, syntheticsMonitorClient) {
    (0, _defineProperty2.default)(this, "debugLog", message => {
      this.serverSetup.logger.debug(`[DeployPrivateLocationMonitors] ${message}`);
    });
    this.serverSetup = serverSetup;
    this.syntheticsMonitorClient = syntheticsMonitorClient;
  }
  async syncPackagePolicies({
    allPrivateLocations,
    encryptedSavedObjects,
    soClient,
    spaceIdToSync
  }) {
    if (allPrivateLocations.length === 0) {
      this.debugLog('No private locations found, skipping sync of private location monitors');
      return;
    }
    const {
      privateLocationAPI
    } = this.syntheticsMonitorClient;
    const {
      configsBySpaces,
      paramsBySpace,
      monitorSpaceIds,
      maintenanceWindows
    } = await this.getAllMonitorConfigs({
      encryptedSavedObjects,
      soClient,
      spaceId: spaceIdToSync
    });
    return this.serverSetup.fleet.runWithCache(async () => {
      this.debugLog(`Starting sync of private location monitors for spaces: ${Array.from(monitorSpaceIds).join(', ')}`);
      for (const spaceId of monitorSpaceIds) {
        var _monitors$length;
        const privateConfigs = [];
        const monitors = configsBySpaces[spaceId];
        this.debugLog(`Processing spaceId: ${spaceId}, monitors count: ${(_monitors$length = monitors === null || monitors === void 0 ? void 0 : monitors.length) !== null && _monitors$length !== void 0 ? _monitors$length : 0}`);
        if (!monitors) {
          continue;
        }
        for (const monitor of monitors) {
          const {
            privateLocations
          } = this.parseLocations(monitor);
          if (privateLocations.length > 0) {
            privateConfigs.push({
              config: monitor,
              globalParams: paramsBySpace[spaceId]
            });
          }
        }
        if (privateConfigs.length > 0) {
          this.debugLog(`Syncing private configs for spaceId: ${spaceId}, privateConfigs count: ${privateConfigs.length}`);
          await privateLocationAPI.editMonitors(privateConfigs, allPrivateLocations, spaceId, maintenanceWindows);
        } else {
          this.debugLog(`No privateConfigs to sync for spaceId: ${spaceId}`);
        }
      }
    });
  }
  async getAllMonitorConfigs({
    soClient,
    encryptedSavedObjects,
    spaceId = _constants.ALL_SPACES_ID
  }) {
    const {
      syntheticsService
    } = this.syntheticsMonitorClient;
    const paramsBySpacePromise = syntheticsService.getSyntheticsParams({
      spaceId
    });
    const maintenanceWindowsPromise = syntheticsService.getMaintenanceWindows(spaceId);
    const monitorConfigRepository = new _monitor_config_repository.MonitorConfigRepository(soClient, encryptedSavedObjects.getClient());
    const monitorsPromise = monitorConfigRepository.findDecryptedMonitors({
      spaceId
    });
    const [paramsBySpace, monitors, maintenanceWindows] = await Promise.all([paramsBySpacePromise, monitorsPromise, maintenanceWindowsPromise]);
    return {
      ...this.mixParamsWithMonitors(monitors, paramsBySpace),
      paramsBySpace,
      maintenanceWindows
    };
  }
  parseLocations(config) {
    const {
      locations
    } = config;
    const privateLocations = locations.filter(loc => !loc.isServiceManaged);
    const publicLocations = locations.filter(loc => loc.isServiceManaged);
    return {
      privateLocations,
      publicLocations
    };
  }
  mixParamsWithMonitors(monitors, paramsBySpace) {
    const configsBySpaces = {};
    const monitorSpaceIds = new Set();
    for (const monitor of monitors) {
      var _monitor$namespaces;
      const spaceId = (_monitor$namespaces = monitor.namespaces) === null || _monitor$namespaces === void 0 ? void 0 : _monitor$namespaces[0];
      if (!spaceId) {
        continue;
      }
      monitorSpaceIds.add(spaceId);
      const normalizedMonitor = (0, _utils.normalizeSecrets)(monitor).attributes;
      const {
        str: paramsString
      } = (0, _format_configs.mixParamsWithGlobalParams)(paramsBySpace[spaceId], normalizedMonitor);
      if (!configsBySpaces[spaceId]) {
        configsBySpaces[spaceId] = [];
      }
      configsBySpaces[spaceId].push((0, _format_configs.formatHeartbeatRequest)({
        spaceId,
        monitor: normalizedMonitor,
        configId: monitor.id
      }, paramsString));
    }
    return {
      configsBySpaces,
      monitorSpaceIds
    };
  }
}
exports.DeployPrivateLocationMonitors = DeployPrivateLocationMonitors;