"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.cleanUpDuplicatedPackagePolicies = cleanUpDuplicatedPackagePolicies;
exports.deleteDuplicatePackagePolicies = deleteDuplicatePackagePolicies;
var _saved_objects = require("../../common/types/saved_objects");
var _synthetics_private_location = require("../synthetics_service/private_location/synthetics_private_location");
var _clean_up_task = require("../synthetics_service/private_location/clean_up_task");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function cleanUpDuplicatedPackagePolicies(serverSetup, soClient, taskState) {
  let performCleanupSync = false;
  const {
    fleet
  } = serverSetup.pluginsStart;
  const {
    logger
  } = serverSetup;
  const debugLog = msg => {
    logger.debug(`[PrivateLocationCleanUpTask] ${msg}`);
  };
  if (taskState.hasAlreadyDoneCleanup) {
    debugLog('Skipping cleanup of duplicated package policies as it has already been done once');
    return {
      performCleanupSync
    };
  } else if (taskState.maxCleanUpRetries <= 0) {
    debugLog('Skipping cleanup of duplicated package policies as max retries have been reached');
    taskState.hasAlreadyDoneCleanup = true;
    taskState.maxCleanUpRetries = 3;
    return {
      performCleanupSync
    };
  }
  debugLog('Starting cleanup of duplicated package policies');
  try {
    var _serverSetup$coreStar, _serverSetup$coreStar2;
    const esClient = (_serverSetup$coreStar = serverSetup.coreStart) === null || _serverSetup$coreStar === void 0 ? void 0 : (_serverSetup$coreStar2 = _serverSetup$coreStar.elasticsearch) === null || _serverSetup$coreStar2 === void 0 ? void 0 : _serverSetup$coreStar2.client.asInternalUser;
    const finder = soClient.createPointInTimeFinder({
      type: _saved_objects.syntheticsMonitorSOTypes,
      fields: ['id', 'name', 'locations', 'origin'],
      namespaces: ['*']
    });
    const privateLocationAPI = new _synthetics_private_location.SyntheticsPrivateLocation(serverSetup);
    const expectedPackagePolicies = new Set();
    for await (const result of finder.find()) {
      result.saved_objects.forEach(monitor => {
        var _monitor$attributes$l;
        (_monitor$attributes$l = monitor.attributes.locations) === null || _monitor$attributes$l === void 0 ? void 0 : _monitor$attributes$l.forEach(location => {
          var _monitor$namespaces;
          const spaceId = (_monitor$namespaces = monitor.namespaces) === null || _monitor$namespaces === void 0 ? void 0 : _monitor$namespaces[0];
          if (!location.isServiceManaged && spaceId) {
            const policyId = privateLocationAPI.getPolicyId({
              origin: monitor.attributes.origin,
              id: monitor.id
            }, location.id, spaceId);
            expectedPackagePolicies.add(policyId);
          }
        });
      });
    }
    finder.close().catch(() => {});
    const packagePoliciesKuery = (0, _clean_up_task.getFilterForTestNowRun)(true);
    const policiesIterator = await fleet.packagePolicyService.fetchAllItemIds(soClient, {
      kuery: packagePoliciesKuery,
      perPage: 100
    });
    const packagePoliciesToDelete = [];
    for await (const packagePoliciesIds of policiesIterator) {
      for (const packagePolicyId of packagePoliciesIds) {
        if (!expectedPackagePolicies.has(packagePolicyId)) {
          packagePoliciesToDelete.push(packagePolicyId);
        }
        // remove it from the set to mark it as found
        expectedPackagePolicies.delete(packagePolicyId);
      }
    }

    // if we have any to delete or any expected that were not found we need to perform a sync
    performCleanupSync = packagePoliciesToDelete.length > 0 || expectedPackagePolicies.size > 0;
    if (packagePoliciesToDelete.length > 0) {
      await deleteDuplicatePackagePolicies(packagePoliciesToDelete, soClient, esClient, serverSetup);
    }
    taskState.hasAlreadyDoneCleanup = true;
    taskState.maxCleanUpRetries = 3;
    return {
      performCleanupSync
    };
  } catch (e) {
    taskState.maxCleanUpRetries -= 1;
    if (taskState.maxCleanUpRetries <= 0) {
      debugLog('Skipping cleanup of duplicated package policies as max retries have been reached');
      taskState.hasAlreadyDoneCleanup = true;
      taskState.maxCleanUpRetries = 3;
    }
    logger.error('[SyncPrivateLocationMonitorsTask] Error cleaning up duplicated package policies', {
      error: e
    });
    return {
      performCleanupSync
    };
  }
}
async function deleteDuplicatePackagePolicies(packagePoliciesToDelete, soClient, esClient, serverSetup) {
  const {
    logger
  } = serverSetup;
  const {
    fleet
  } = serverSetup.pluginsStart;
  logger.info(`[PrivateLocationCleanUpTask] Found ${packagePoliciesToDelete.length} duplicate package policies to delete.`);
  // Delete it in batches of 100 to avoid sending too large payloads at once.
  const BATCH_SIZE = 100;
  const total = packagePoliciesToDelete.length;
  const totalBatches = Math.ceil(total / BATCH_SIZE);
  for (let i = 0; i < total; i += BATCH_SIZE) {
    const batch = packagePoliciesToDelete.slice(i, i + BATCH_SIZE);
    const batchIndex = Math.floor(i / BATCH_SIZE) + 1;
    logger.info(`[PrivateLocationCleanUpTask] Deleting batch ${batchIndex}/${totalBatches} (size=${batch.length}), with ids [${batch.join(`, `)}]`);
    await fleet.packagePolicyService.delete(soClient, esClient, batch, {
      force: true
    });
  }
}